<?php
require_once __DIR__ . '/../lib/referrals.php';
ini_set('display_errors', '1');
error_reporting(E_ALL);

class FakeResult {
    private $rows;
    private $index = 0;

    public function __construct(array $rows) {
        $this->rows = array_values($rows);
    }

    public function fetch_assoc() {
        if ($this->index >= count($this->rows)) {
            return null;
        }
        return $this->rows[$this->index++];
    }
}

class FakeMySQLi {
    public $settings = [];
    public $users = [];
    public $orders = [];
    public $referralRewards = [];
    public $discountCodes = [];
    public $affected_rows = 0;
    public $insert_id = 0;
    private $transactionBackup = null;
    private $lastDiscountId = 0;

    public function begin_transaction(int $flags = 0, ?string $name = null): bool
    {
        $this->transactionBackup = [
            'referralRewards' => $this->referralRewards,
            'discountCodes' => $this->discountCodes,
        ];
        return true;
    }

    public function commit(int $flags = 0, ?string $name = null): bool
    {
        $this->transactionBackup = null;
        return true;
    }

    public function rollback(int $flags = 0, ?string $name = null): bool
    {
        if ($this->transactionBackup !== null) {
            $this->referralRewards = $this->transactionBackup['referralRewards'];
            $this->discountCodes = $this->transactionBackup['discountCodes'];
            $this->transactionBackup = null;
        }
        return true;
    }

    public function prepare($query, $options = null): FakeStmt
    {
        return new FakeStmt($this, $query);
    }

    public function query($query, $resultmode = null)
    {
        return true;
    }

    public function nextDiscountId(): int
    {
        $this->lastDiscountId += 1;
        $this->setInsertId($this->lastDiscountId);
        return $this->lastDiscountId;
    }

    public function setAffectedRows(int $value): void
    {
        $this->affected_rows = $value;
    }

    public function setInsertId(int $value): void
    {
        $this->insert_id = $value;
    }
}

class FakeStmt {
    private $conn;
    private $sql;
    private $params = [];
    private $resultRows = [];
    private $affectedRowsInternal = 0;
    private $insertIdInternal = 0;

    public function __construct(FakeMySQLi $conn, string $sql)
    {
        $this->conn = $conn;
        $this->sql = trim($sql);
    }

    public function bind_param($types, ...$vars): bool
    {
        $this->params = [];
        if (count($vars) === 0 && func_num_args() > 1) {
            $vars = array_slice(func_get_args(), 1);
        }
        foreach ($vars as $index => &$value) {
            $this->params[$index] = $value;
        }
        return true;
    }

    private function applyAffectedRows(int $value): void
    {
        $this->affectedRowsInternal = $value;
        $this->conn->setAffectedRows($value);
    }

    private function applyInsertId(int $value): void
    {
        $this->insertIdInternal = $value;
        $this->conn->setInsertId($value);
    }

    public function execute(?array $params = null): bool
    {
        if (is_array($params)) {
            $this->params = array_values($params);
        }
        $sql = preg_replace('/\s+/', ' ', $this->sql);
        $this->applyAffectedRows(0);
        $this->resultRows = [];
        $this->applyInsertId(0);

        if (stripos($sql, 'SELECT value FROM settings') === 0) {
            $key = $this->params[0] ?? '';
            if (array_key_exists($key, $this->conn->settings)) {
                $this->resultRows = [['value' => $this->conn->settings[$key]]];
            }
        } elseif (stripos($sql, 'SELECT `key`, `value` FROM settings') === 0) {
            $rows = [];
            foreach ($this->params as $key) {
                if (array_key_exists($key, $this->conn->settings)) {
                    $rows[] = ['key' => $key, 'value' => $this->conn->settings[$key]];
                }
            }
            $this->resultRows = $rows;
        } elseif (stripos($sql, 'SELECT referral_code FROM user WHERE id') === 0) {
            $id = (int)($this->params[0] ?? 0);
            $user = $this->conn->users[$id] ?? null;
            if ($user) {
                $this->resultRows = [['referral_code' => $user['referral_code'] ?? null]];
            }
        } elseif (stripos($sql, 'SELECT id FROM user WHERE referral_code') === 0) {
            $code = $this->params[0] ?? '';
            foreach ($this->conn->users as $user) {
                if (($user['referral_code'] ?? null) === $code) {
                    $this->resultRows = [['id' => $user['id']]];
                    break;
                }
            }
        } elseif (stripos($sql, 'UPDATE user SET referred_by') === 0) {
            $ref = (int)($this->params[0] ?? 0);
            $id = (int)($this->params[1] ?? 0);
            if (isset($this->conn->users[$id]) && empty($this->conn->users[$id]['referred_by'])) {
                $this->conn->users[$id]['referred_by'] = $ref;
                $this->applyAffectedRows(1);
            }
        } elseif (stripos($sql, 'SELECT referred_by FROM user WHERE id') === 0) {
            $id = (int)($this->params[0] ?? 0);
            $user = $this->conn->users[$id] ?? null;
            if ($user) {
                $this->resultRows = [['referred_by' => $user['referred_by'] ?? null]];
            }
        } elseif (stripos($sql, 'SELECT id, referred_by FROM user WHERE id') === 0) {
            $id = (int)($this->params[0] ?? 0);
            $user = $this->conn->users[$id] ?? null;
            if ($user) {
                $this->resultRows = [['id' => $user['id'], 'referred_by' => $user['referred_by'] ?? null]];
            }
        } elseif (stripos($sql, 'SELECT COUNT(*) AS cnt FROM orders') === 0) {
            $userId = (int)($this->params[0] ?? 0);
            $count = 0;
            foreach ($this->conn->orders as $order) {
                if ((int)$order['user_id'] === $userId && ($order['status'] ?? '') === 'approved') {
                    $count++;
                }
            }
            $this->resultRows = [['cnt' => $count]];
        } elseif (stripos($sql, 'SELECT id FROM referral_rewards WHERE referrer_id') === 0) {
            $ref = (int)($this->params[0] ?? 0);
            $invitee = (int)($this->params[1] ?? 0);
            foreach ($this->conn->referralRewards as $reward) {
                if ((int)$reward['referrer_id'] === $ref && (int)$reward['invitee_id'] === $invitee) {
                    $this->resultRows = [['id' => $reward['id']]];
                    break;
                }
            }
        } elseif (stripos($sql, 'INSERT INTO discount_codes') === 0) {
            $code = (string)($this->params[0] ?? '');
            $owner = (int)($this->params[1] ?? 0);
            $type = (string)($this->params[2] ?? 'fixed');
            $value = (float)($this->params[3] ?? 0);
            $maxUses = (int)($this->params[4] ?? 1);
            $perUser = (int)($this->params[5] ?? 1);
            $minOrderTotal = (float)($this->params[6] ?? 0);
            $minOrderAmount = (float)($this->params[7] ?? 0);
            $expiresAt = $this->params[8] ?? null;
            $note = $this->params[9] ?? null;
            $meta = $this->params[10] ?? null;
            $id = $this->conn->nextDiscountId();
            $this->conn->discountCodes[] = [
                'id' => $id,
                'code' => $code,
                'owner_user_id' => $owner,
                'type' => $type,
                'value' => $value,
                'max_uses' => $maxUses,
                'per_user_limit' => $perUser,
                'min_order_total' => $minOrderTotal,
                'min_order_amount' => $minOrderAmount,
                'expires_at' => $expiresAt,
                'note' => $note,
                'meta' => $meta,
            ];
            $this->applyInsertId($id);
            $this->applyAffectedRows(1);
        } elseif (stripos($sql, 'SELECT id FROM discount_codes WHERE code') === 0) {
            $code = $this->params[0] ?? '';
            foreach ($this->conn->discountCodes as $row) {
                if ($row['code'] === $code) {
                    $this->resultRows = [['id' => $row['id']]];
                    break;
                }
            }
        } elseif (stripos($sql, 'INSERT IGNORE INTO referral_rewards') === 0) {
            $ref = (int)($this->params[0] ?? 0);
            $invitee = (int)($this->params[1] ?? 0);
            $orderId = (int)($this->params[2] ?? 0);
            $code = (string)($this->params[3] ?? '');
            foreach ($this->conn->referralRewards as $reward) {
                if ((int)$reward['referrer_id'] === $ref && (int)$reward['invitee_id'] === $invitee) {
                    return true;
                }
            }
            $this->conn->referralRewards[] = [
                'id' => count($this->conn->referralRewards) + 1,
                'referrer_id' => $ref,
                'invitee_id' => $invitee,
                'order_id' => $orderId,
                'code' => $code,
            ];
            $this->applyAffectedRows(1);
        } elseif (stripos($sql, 'SELECT COUNT(*) AS cnt FROM user WHERE referred_by') === 0) {
            $ref = (int)($this->params[0] ?? 0);
            $count = 0;
            foreach ($this->conn->users as $user) {
                if ((int)($user['referred_by'] ?? 0) === $ref) {
                    $count++;
                }
            }
            $this->resultRows = [['cnt' => $count]];
        } elseif (stripos($sql, 'SELECT COUNT(*) AS cnt FROM referral_rewards WHERE referrer_id') === 0) {
            $ref = (int)($this->params[0] ?? 0);
            $count = 0;
            foreach ($this->conn->referralRewards as $reward) {
                if ((int)$reward['referrer_id'] === $ref) {
                    $count++;
                }
            }
            $this->resultRows = [['cnt' => $count]];
        }

        return true;
    }

    public function get_result()
    {
        return new FakeResult($this->resultRows);
    }

    public function __get($name)
    {
        if ($name === 'affected_rows') {
            return $this->affectedRowsInternal;
        }
        if ($name === 'insert_id') {
            return $this->insertIdInternal;
        }
        return null;
    }

    public function close(): true
    {
        $this->params = [];
        $this->resultRows = [];
        return true;
    }
}

function assertTrue($condition, string $message): void {
    if (!$condition) {
        throw new RuntimeException($message);
    }
}

function assertEquals($expected, $actual, string $message): void {
    if ($expected !== $actual) {
        throw new RuntimeException($message . ' Expected ' . var_export($expected, true) . ' got ' . var_export($actual, true));
    }
}

function make_connection(array $settings = []): FakeMySQLi {
    $conn = new FakeMySQLi();
    $conn->settings = $settings;
    return $conn;
}

function with_sender(array &$sent, callable $callback): void {
    $GLOBALS['__referral_test_sender'] = function ($chatId, $text, $options) use (&$sent) {
        $sent[] = ['chat_id' => $chatId, 'text' => $text, 'options' => $options];
        return true;
    };
    try {
        $callback();
    } finally {
        unset($GLOBALS['__referral_test_sender']);
    }
}

function test_assign_referrer_once(): void {
    $conn = make_connection(['referral_status' => 'on']);
    $conn->users = [
        1 => ['id' => 1, 'referral_code' => 'AAA', 'referred_by' => null],
        2 => ['id' => 2, 'referral_code' => 'BBB', 'referred_by' => null],
    ];

    $first = referral_assign_referred_by($conn, 2, 1);
    $second = referral_assign_referred_by($conn, 2, 1);
    assertTrue($first === true, 'First referral assignment should succeed');
    assertTrue($second === false, 'Second referral assignment should be ignored');
    assertEquals(1, $conn->users[2]['referred_by'], 'User should remain referred by first referrer');
}

function base_reward_settings(): array {
    return [
        'referral_status' => 'on',
        'referral_reward_type' => 'fixed',
        'referral_reward_value' => '10000',
        'referral_min_order_total' => '0',
        'referral_code_ttl_days' => '7',
        'referral_reward_prefix' => 'TST',
        'referral_reward_limit' => '1',
    ];
}

function test_reward_on_first_order(): void {
    $conn = make_connection(base_reward_settings());
    $conn->users = [
        10 => ['id' => 10, 'referral_code' => 'AAA111', 'referred_by' => null],
        20 => ['id' => 20, 'referral_code' => 'BBB222', 'referred_by' => 10],
    ];
    $conn->orders = [
        ['id' => 501, 'user_id' => 20, 'status' => 'approved'],
    ];
    $order = ['id' => 501, 'user_id' => 20, 'status' => 'approved', 'type' => 'new'];
    $sent = [];
    with_sender($sent, function () use ($conn, $order) {
        referral_process_successful_order($conn, $order);
    });

    assertEquals(1, count($conn->discountCodes), 'One discount code should be created');
    assertEquals(1, count($conn->referralRewards), 'One referral reward should be recorded');
    assertEquals(1, count($sent), 'Referrer should receive one message');
    assertTrue(strpos($sent[0]['text'], 'کد تخفیف') !== false, 'Notification text should mention discount code');
}

function test_no_reward_on_second_order(): void {
    $conn = make_connection(base_reward_settings());
    $conn->users = [
        10 => ['id' => 10, 'referral_code' => 'AAA111', 'referred_by' => null],
        20 => ['id' => 20, 'referral_code' => 'BBB222', 'referred_by' => 10],
    ];
    $conn->orders = [
        ['id' => 501, 'user_id' => 20, 'status' => 'approved'],
        ['id' => 502, 'user_id' => 20, 'status' => 'approved'],
    ];
    $order = ['id' => 502, 'user_id' => 20, 'status' => 'approved', 'type' => 'new'];
    $sent = [];
    with_sender($sent, function () use ($conn, $order) {
        referral_process_successful_order($conn, $order);
    });

    assertEquals(0, count($conn->discountCodes), 'No new discount code on second order');
    assertEquals(0, count($conn->referralRewards), 'No reward recorded on second order');
    assertEquals(0, count($sent), 'No notification should be sent');
}

function test_no_reward_for_renewal(): void {
    $conn = make_connection(base_reward_settings());
    $conn->users = [
        10 => ['id' => 10, 'referred_by' => null],
        20 => ['id' => 20, 'referred_by' => 10],
    ];
    $conn->orders = [
        ['id' => 501, 'user_id' => 20, 'status' => 'approved'],
    ];
    $order = ['id' => 501, 'user_id' => 20, 'status' => 'approved', 'type' => 'exten'];
    $sent = [];
    with_sender($sent, function () use ($conn, $order) {
        referral_process_successful_order($conn, $order);
    });
    assertEquals(0, count($conn->discountCodes), 'Renewal should not generate code');
}

function test_no_reward_when_disabled(): void {
    $conn = make_connection(['referral_status' => 'off']);
    $conn->users = [
        1 => ['id' => 1, 'referred_by' => null],
        2 => ['id' => 2, 'referred_by' => 1],
    ];
    $conn->orders = [
        ['id' => 1, 'user_id' => 2, 'status' => 'approved'],
    ];
    $order = ['id' => 1, 'user_id' => 2, 'status' => 'approved', 'type' => 'new'];
    referral_process_successful_order($conn, $order);
    assertEquals(0, count($conn->discountCodes), 'Disabled referral program should not create codes');
}

function test_idempotent_processing(): void {
    $conn = make_connection(base_reward_settings());
    $conn->users = [
        10 => ['id' => 10, 'referred_by' => null],
        20 => ['id' => 20, 'referred_by' => 10],
    ];
    $conn->orders = [
        ['id' => 701, 'user_id' => 20, 'status' => 'approved'],
    ];
    $order = ['id' => 701, 'user_id' => 20, 'status' => 'approved', 'type' => 'new'];
    $sent = [];
    with_sender($sent, function () use ($conn, $order) {
        referral_process_successful_order($conn, $order);
        referral_process_successful_order($conn, $order);
    });
    assertEquals(1, count($conn->discountCodes), 'Duplicate processing must not create extra codes');
    assertEquals(1, count($conn->referralRewards), 'Duplicate processing must not create extra rewards');
}

function run_tests(): void {
    $tests = [
        'test_assign_referrer_once',
        'test_reward_on_first_order',
        'test_no_reward_on_second_order',
        'test_no_reward_for_renewal',
        'test_no_reward_when_disabled',
        'test_idempotent_processing',
    ];
    foreach ($tests as $test) {
        $test();
    }
    echo "All referral tests passed\n";
}

run_tests();
