<?php
if (!function_exists('sm_jdate_safe')) {
    function sm_jdate_safe(string $format, int $timestamp): string {
        if (function_exists('jdate')) {
            return jdate($format, $timestamp);
        }
        return date($format, $timestamp);
    }
}

if (!function_exists('ensure_server_metadata_schema')) {
    function ensure_server_metadata_schema(mysqli $connect): void {
        $connect->query("CREATE TABLE IF NOT EXISTS server_tags (
            id INT AUTO_INCREMENT PRIMARY KEY,
            name VARCHAR(64) NOT NULL,
            color VARCHAR(16) DEFAULT '#6366f1',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci");

        $connect->query("CREATE TABLE IF NOT EXISTS server_tag_map (
            server_id INT NOT NULL,
            tag_id INT NOT NULL,
            PRIMARY KEY(server_id, tag_id)
        ) CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci");

        try { $connect->query("ALTER TABLE server_tag_map ADD INDEX idx_tag (tag_id)"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE server_tag_map ADD CONSTRAINT fk_server_tag_server FOREIGN KEY (server_id) REFERENCES servers(id) ON DELETE CASCADE"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE server_tag_map ADD CONSTRAINT fk_server_tag_tag FOREIGN KEY (tag_id) REFERENCES server_tags(id) ON DELETE CASCADE"); } catch (Throwable $e) {}

        try { $connect->query("ALTER TABLE servers ADD COLUMN region VARCHAR(64) NULL"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE servers ADD COLUMN max_capacity INT NULL DEFAULT 0"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE servers ADD COLUMN current_load INT NULL DEFAULT 0"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE servers ADD COLUMN maintenance_mode TINYINT(1) NOT NULL DEFAULT 0"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE servers ADD COLUMN maintenance_note VARCHAR(255) NULL"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE servers ADD COLUMN marzneshin_service_ids VARCHAR(255) NULL"); } catch (Throwable $e) {}

        try { $connect->query("ALTER TABLE orders ADD COLUMN created_at TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE orders ADD COLUMN status VARCHAR(32) NOT NULL DEFAULT 'pending'"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE user ADD COLUMN created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP"); } catch (Throwable $e) {}
    }
}

if (!function_exists('fetch_server_tags')) {
    function fetch_server_tags(mysqli $connect): array {
        $sql = "SELECT st.*, COUNT(stm.server_id) AS usage_count
                FROM server_tags st
                LEFT JOIN server_tag_map stm ON stm.tag_id = st.id
                GROUP BY st.id
                ORDER BY st.name ASC";
        $result = $connect->query($sql);
        $tags = [];
        if ($result) {
            while ($row = $result->fetch_assoc()) {
                $row['usage_count'] = (int)($row['usage_count'] ?? 0);
                $tags[] = $row;
            }
            $result->close();
        }
        return $tags;
    }
}

if (!function_exists('sync_server_tags')) {
    function sync_server_tags(mysqli $connect, int $server_id, array $tag_ids): void {
        $stmt = $connect->prepare("DELETE FROM server_tag_map WHERE server_id = ?");
        $stmt->bind_param('i', $server_id);
        $stmt->execute();
        $stmt->close();

        if (empty($tag_ids)) {
            return;
        }

        $stmt_insert = $connect->prepare("INSERT IGNORE INTO server_tag_map (server_id, tag_id) VALUES (?, ?)");
        foreach ($tag_ids as $tag_id) {
            $tag_id = (int)$tag_id;
            if ($tag_id <= 0) {
                continue;
            }
            $stmt_insert->bind_param('ii', $server_id, $tag_id);
            $stmt_insert->execute();
        }
        $stmt_insert->close();
    }
}

if (!function_exists('fetch_servers_with_metadata')) {
    function fetch_servers_with_metadata(mysqli $connect): array {
        $sql = "SELECT s.*, 
                       GROUP_CONCAT(st.name ORDER BY st.name SEPARATOR ',') AS tag_names,
                       GROUP_CONCAT(st.color ORDER BY st.name SEPARATOR ',') AS tag_colors,
                       GROUP_CONCAT(st.id ORDER BY st.name SEPARATOR ',') AS tag_ids
                FROM servers s
                LEFT JOIN server_tag_map stm ON s.id = stm.server_id
                LEFT JOIN server_tags st ON st.id = stm.tag_id
                GROUP BY s.id
                ORDER BY s.id ASC";
        $result = $connect->query($sql);
        $servers = [];
        if ($result) {
            while ($row = $result->fetch_assoc()) {
                $tag_ids = array_filter(array_map('intval', explode(',', (string)($row['tag_ids'] ?? ''))));
                $tag_names = array_filter(explode(',', (string)($row['tag_names'] ?? '')));
                $tag_colors = array_filter(explode(',', (string)($row['tag_colors'] ?? '')));
                $tags = [];
                foreach ($tag_ids as $index => $tag_id) {
                    $tags[] = [
                        'id' => $tag_id,
                        'name' => $tag_names[$index] ?? '',
                        'color' => $tag_colors[$index] ?? '#6366f1',
                    ];
                }
                $row['tags'] = $tags;
                unset($row['tag_ids'], $row['tag_names'], $row['tag_colors']);

                $max_capacity = isset($row['max_capacity']) ? (int)$row['max_capacity'] : 0;
                $current_load = isset($row['current_load']) ? (int)$row['current_load'] : 0;
                $utilization = ($max_capacity > 0) ? round(($current_load / $max_capacity) * 100) : null;
                $row['max_capacity'] = $max_capacity;
                $row['current_load'] = $current_load;
                $row['utilization'] = $utilization;

                if ((int)($row['maintenance_mode'] ?? 0) === 1) {
                    $row['health_status'] = 'maintenance';
                } elseif (($row['status'] ?? '') === 'on') {
                    $row['health_status'] = ($utilization !== null && $utilization >= 85) ? 'warning' : 'online';
                } else {
                    $row['health_status'] = 'offline';
                }

                $servers[] = $row;
            }
            $result->close();
        }
        return $servers;
    }
}

if (!function_exists('compute_extended_stats')) {
    function compute_extended_stats(mysqli $connect): array {
        $now = time();
        $stats = [
            'active_users' => 0,
            'expiring_week' => 0,
            'active_services_total' => 0,
            'expired_services_total' => 0,
            'pending_orders' => 0,
            'revenue_30_days' => 0,
            'revenue_growth' => null,
            'avg_order_amount' => 0,
            'payment_breakdown' => [],
            'daily_trend' => [],
            'top_servers' => [],
            'server_health' => [
                'online' => 0,
                'offline' => 0,
                'maintenance' => 0,
                'average_utilization' => null,
                'total_capacity' => 0,
                'total_load' => 0,
            ],
            'new_users_7_days' => 0,
            'retention' => [
                'period_days' => 30,
                'new_customers' => 0,
                'returning_customers' => 0,
                'active_customers' => 0,
                'repeat_rate' => 0,
                'average_orders_per_customer' => 0,
                'dormant_customers' => 0,
            ],
            'region_breakdown' => [],
        ];

        $res = $connect->query("SELECT COUNT(DISTINCT owner) AS c FROM services WHERE expire > {$now}");
        if ($res) {
            $stats['active_users'] = (int)($res->fetch_assoc()['c'] ?? 0);
            $res->close();
        }

        $res = $connect->query("SELECT COUNT(*) AS c FROM services WHERE expire BETWEEN {$now} AND " . ($now + 7 * 86400));
        if ($res) {
            $stats['expiring_week'] = (int)($res->fetch_assoc()['c'] ?? 0);
            $res->close();
        }

        $res = $connect->query("SELECT COUNT(*) AS c FROM services WHERE expire > {$now}");
        if ($res) {
            $stats['active_services_total'] = (int)($res->fetch_assoc()['c'] ?? 0);
            $res->close();
        }

        $res = $connect->query("SELECT COUNT(*) AS c FROM services WHERE expire <= {$now}");
        if ($res) {
            $stats['expired_services_total'] = (int)($res->fetch_assoc()['c'] ?? 0);
            $res->close();
        }

        $res = $connect->query("SELECT COUNT(*) AS c FROM orders WHERE status = 'pending'");
        if ($res) {
            $stats['pending_orders'] = (int)($res->fetch_assoc()['c'] ?? 0);
            $res->close();
        }

        $res = $connect->query("SELECT COUNT(*) AS c FROM user WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)");
        if ($res) {
            $stats['new_users_7_days'] = (int)($res->fetch_assoc()['c'] ?? 0);
            $res->close();
        }

        $res = $connect->query("SELECT COALESCE(SUM(amount),0) AS total FROM orders WHERE status='approved' AND created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)");
        if ($res) {
            $stats['revenue_30_days'] = (int)($res->fetch_assoc()['total'] ?? 0);
            $res->close();
        }

        $res = $connect->query("SELECT COALESCE(SUM(amount),0) AS total FROM orders WHERE status='approved' AND created_at >= DATE_SUB(NOW(), INTERVAL 60 DAY) AND created_at < DATE_SUB(NOW(), INTERVAL 30 DAY)");
        if ($res) {
            $prev = (int)($res->fetch_assoc()['total'] ?? 0);
            if ($prev > 0) {
                $stats['revenue_growth'] = round((($stats['revenue_30_days'] - $prev) / $prev) * 100, 1);
            } elseif ($stats['revenue_30_days'] > 0) {
                $stats['revenue_growth'] = 100.0;
            } else {
                $stats['revenue_growth'] = 0.0;
            }
            $res->close();
        }

        $res = $connect->query("SELECT COALESCE(AVG(amount),0) AS avg_amount FROM orders WHERE status='approved' AND created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)");
        if ($res) {
            $stats['avg_order_amount'] = (float)($res->fetch_assoc()['avg_amount'] ?? 0);
            $res->close();
        }

        $res = $connect->query("SELECT COALESCE(type, 'unknown') AS payment_type, COUNT(*) AS orders_count, COALESCE(SUM(amount),0) AS total_amount
                                 FROM orders WHERE status='approved'
                                 GROUP BY payment_type");
        if ($res) {
            while ($row = $res->fetch_assoc()) {
                $stats['payment_breakdown'][] = [
                    'type' => $row['payment_type'],
                    'orders_count' => (int)($row['orders_count'] ?? 0),
                    'total_amount' => (int)($row['total_amount'] ?? 0),
                ];
            }
            $res->close();
        }

        $res = $connect->query("SELECT DATE(IFNULL(created_at, NOW())) AS order_day, COUNT(*) AS orders_count, COALESCE(SUM(amount),0) AS total_amount
                                 FROM orders
                                 WHERE status='approved' AND DATE(IFNULL(created_at, NOW())) >= DATE_SUB(CURDATE(), INTERVAL 14 DAY)
                                 GROUP BY order_day
                                 ORDER BY order_day ASC");
        if ($res) {
            while ($row = $res->fetch_assoc()) {
                $dayTimestamp = strtotime($row['order_day']);
                $stats['daily_trend'][] = [
                    'day' => sm_jdate_safe('Y/m/d', $dayTimestamp ?: time()),
                    'orders_count' => (int)($row['orders_count'] ?? 0),
                    'total_amount' => (int)($row['total_amount'] ?? 0),
                ];
            }
            $res->close();
        }

        $servers = fetch_servers_with_metadata($connect);
        $utilSum = 0;
        $utilCount = 0;
        foreach ($servers as $server) {
            if ((int)($server['maintenance_mode'] ?? 0) === 1) {
                $stats['server_health']['maintenance']++;
            } elseif (($server['status'] ?? '') === 'on') {
                $stats['server_health']['online']++;
            } else {
                $stats['server_health']['offline']++;
            }
            $stats['server_health']['total_capacity'] += (int)($server['max_capacity'] ?? 0);
            $stats['server_health']['total_load'] += (int)($server['current_load'] ?? 0);
            if ($server['utilization'] !== null) {
                $utilSum += (int)$server['utilization'];
                $utilCount++;
            }
        }
        if ($utilCount > 0) {
            $stats['server_health']['average_utilization'] = round($utilSum / $utilCount);
        }

        $sqlTop = "SELECT sv.id, sv.display_name, sv.region, sv.max_capacity, sv.current_load,
                          COUNT(s.id) AS active_services
                   FROM servers sv
                   LEFT JOIN services s ON sv.name = s.svname AND s.expire > {$now}
                   GROUP BY sv.id
                   ORDER BY active_services DESC
                   LIMIT 5";
        $res = $connect->query($sqlTop);
        if ($res) {
            while ($row = $res->fetch_assoc()) {
                $maxCapacity = (int)($row['max_capacity'] ?? 0);
                $currentLoad = (int)($row['current_load'] ?? 0);
                $row['utilization'] = ($maxCapacity > 0) ? round(($currentLoad / max(1, $maxCapacity)) * 100) : null;
                $row['active_services'] = (int)($row['active_services'] ?? 0);
                $stats['top_servers'][] = $row;
            }
            $res->close();
        }

        $stats['retention'] = compute_customer_retention_stats($connect, 30);
        $stats['region_breakdown'] = compute_region_capacity_breakdown($connect);

        return $stats;
    }
}

if (!function_exists('fetch_order_timeline')) {
    function fetch_order_timeline(mysqli $connect, int $limit = 15): array {
        $limit = max(1, min(50, $limit));
        $sql = "SELECT order_code, user_id, amount, stars_amount, status, type, location, created_at
                FROM orders
                ORDER BY IFNULL(created_at, NOW()) DESC, order_code DESC
                LIMIT {$limit}";
        $result = $connect->query($sql);
        $timeline = [];
        if ($result) {
            while ($row = $result->fetch_assoc()) {
                $timestamp = isset($row['created_at']) ? strtotime($row['created_at']) : time();
                $timeline[] = [
                    'order_code' => $row['order_code'] ?? '-',
                    'user_id' => $row['user_id'] ?? '-',
                    'amount' => (int)($row['amount'] ?? 0),
                    'stars_amount' => (int)($row['stars_amount'] ?? 0),
                    'status' => $row['status'] ?? 'pending',
                    'type' => $row['type'] ?? '-',
                    'location' => $row['location'] ?? '-',
                    'created_at' => $timestamp,
                    'created_at_jalali' => sm_jdate_safe('Y/m/d H:i', $timestamp ?: time()),
                ];
            }
            $result->close();
        }
        return $timeline;
    }
}

if (!function_exists('compute_customer_retention_stats')) {
    function compute_customer_retention_stats(mysqli $connect, int $days = 30): array {
        $days = max(1, $days);

        $stats = [
            'period_days' => $days,
            'new_customers' => 0,
            'returning_customers' => 0,
            'active_customers' => 0,
            'repeat_rate' => 0,
            'average_orders_per_customer' => 0,
            'dormant_customers' => 0,
        ];

        $sql = "SELECT COUNT(DISTINCT user_id) AS total_customers FROM orders WHERE status='approved'";
        $res = $connect->query($sql);
        $totalCustomers = 0;
        if ($res) {
            $totalCustomers = (int)($res->fetch_assoc()['total_customers'] ?? 0);
            $res->close();
        }

        $sqlRetention = "SELECT
                SUM(CASE WHEN first_order >= DATE_SUB(NOW(), INTERVAL ? DAY) THEN 1 ELSE 0 END) AS new_customers,
                SUM(CASE WHEN first_order < DATE_SUB(NOW(), INTERVAL ? DAY) THEN 1 ELSE 0 END) AS returning_customers
            FROM (
                SELECT user_id,
                       MIN(IFNULL(created_at, NOW())) AS first_order,
                       MAX(IFNULL(created_at, NOW())) AS last_order
                FROM orders
                WHERE status='approved'
                GROUP BY user_id
            ) AS t
            WHERE last_order >= DATE_SUB(NOW(), INTERVAL ? DAY)";

        $stmt = $connect->prepare($sqlRetention);
        if ($stmt) {
            $stmt->bind_param('iii', $days, $days, $days);
            if ($stmt->execute()) {
                $result = $stmt->get_result();
                if ($result) {
                    $row = $result->fetch_assoc();
                    $stats['new_customers'] = (int)($row['new_customers'] ?? 0);
                    $stats['returning_customers'] = (int)($row['returning_customers'] ?? 0);
                    $result->close();
                }
            }
            $stmt->close();
        }

        $activeCustomers = $stats['new_customers'] + $stats['returning_customers'];
        $stats['active_customers'] = $activeCustomers;

        $sqlOrders = "SELECT COUNT(*) AS total_orders, COUNT(DISTINCT user_id) AS active_users
            FROM orders
            WHERE status='approved' AND IFNULL(created_at, NOW()) >= DATE_SUB(NOW(), INTERVAL {$days} DAY)";
        $resOrders = $connect->query($sqlOrders);
        $totalOrders = 0;
        $activeUsers = 0;
        if ($resOrders) {
            $row = $resOrders->fetch_assoc();
            $totalOrders = (int)($row['total_orders'] ?? 0);
            $activeUsers = (int)($row['active_users'] ?? 0);
            $resOrders->close();
        }

        if ($activeCustomers > 0) {
            $stats['repeat_rate'] = round(($stats['returning_customers'] / $activeCustomers) * 100, 1);
        }
        if ($activeUsers > 0) {
            $stats['average_orders_per_customer'] = round($totalOrders / max(1, $activeUsers), 2);
        }

        $dormant = $totalCustomers - $activeCustomers;
        $stats['dormant_customers'] = ($dormant > 0) ? $dormant : 0;

        return $stats;
    }
}

if (!function_exists('compute_region_capacity_breakdown')) {
    function compute_region_capacity_breakdown(mysqli $connect): array {
        $sql = "SELECT
                CASE WHEN region IS NULL OR region = '' THEN 'نامشخص' ELSE region END AS region_name,
                COUNT(*) AS server_count,
                SUM(COALESCE(max_capacity, 0)) AS total_capacity,
                SUM(COALESCE(current_load, 0)) AS total_load,
                SUM(CASE WHEN maintenance_mode = 1 THEN 1 ELSE 0 END) AS maintenance_count,
                SUM(CASE WHEN status = 'on' THEN 1 ELSE 0 END) AS online_count
            FROM servers
            GROUP BY region_name
            ORDER BY total_capacity DESC, region_name ASC";

        $result = $connect->query($sql);
        $regions = [];
        if ($result) {
            while ($row = $result->fetch_assoc()) {
                $capacity = (int)($row['total_capacity'] ?? 0);
                $load = (int)($row['total_load'] ?? 0);
                $regions[] = [
                    'region' => $row['region_name'],
                    'server_count' => (int)($row['server_count'] ?? 0),
                    'online_count' => (int)($row['online_count'] ?? 0),
                    'maintenance_count' => (int)($row['maintenance_count'] ?? 0),
                    'total_capacity' => $capacity,
                    'total_load' => $load,
                    'available_capacity' => ($capacity > $load) ? ($capacity - $load) : 0,
                    'utilization' => ($capacity > 0) ? round(($load / max(1, $capacity)) * 100) : null,
                ];
            }
            $result->close();
        }
        return $regions;
    }
}
