<?php
// Discount codes library: schema + helpers

require_once __DIR__ . '/../config.php';

function ensure_discount_schema(mysqli $connect): void {
    // discount_codes table
    $connect->query(
        "CREATE TABLE IF NOT EXISTS discount_codes (
            id INT AUTO_INCREMENT PRIMARY KEY,
            code VARCHAR(64) NOT NULL UNIQUE,
            owner_user_id BIGINT NULL,
            type ENUM('percent','fixed') NOT NULL DEFAULT 'percent',
            value DECIMAL(10,2) NOT NULL DEFAULT 0,
            max_uses INT NULL DEFAULT NULL,
            used_count INT NOT NULL DEFAULT 0,
            per_user_limit INT NULL DEFAULT NULL,
            min_order_total DECIMAL(10,2) NOT NULL DEFAULT 0,
            min_order_amount DECIMAL(10,2) NULL DEFAULT 0,
            expires_at DATETIME NULL DEFAULT NULL,
            is_active TINYINT(1) NOT NULL DEFAULT 1,
            note VARCHAR(255) NULL DEFAULT NULL,
            meta TEXT NULL,
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci"
    );

    try { $connect->query("ALTER TABLE discount_codes ADD COLUMN owner_user_id BIGINT NULL AFTER code"); } catch (Throwable $e) {}
    try { $connect->query("ALTER TABLE discount_codes ADD COLUMN min_order_total DECIMAL(10,2) NOT NULL DEFAULT 0"); } catch (Throwable $e) {}
    try { $connect->query("ALTER TABLE discount_codes MODIFY COLUMN min_order_amount DECIMAL(10,2) NULL DEFAULT 0"); } catch (Throwable $e) {}
    try { $connect->query("ALTER TABLE discount_codes ADD COLUMN meta TEXT NULL"); } catch (Throwable $e) {}
    try { $connect->query("ALTER TABLE discount_codes ADD INDEX idx_owner_user (owner_user_id)"); } catch (Throwable $e) {}

    // discount_usages table
    $connect->query(
        "CREATE TABLE IF NOT EXISTS discount_usages (
            id INT AUTO_INCREMENT PRIMARY KEY,
            discount_id INT NOT NULL,
            order_code VARCHAR(64) NOT NULL,
            user_id BIGINT NOT NULL,
            is_confirmed TINYINT(1) NOT NULL DEFAULT 0,
            used_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            UNIQUE KEY uq_disc_user_order (discount_id, user_id, order_code),
            INDEX idx_order_code (order_code),
            CONSTRAINT fk_discount_usages_discount_id FOREIGN KEY (discount_id)
                REFERENCES discount_codes(id) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci"
    );

    // orders table alterations (safe try-add)
    try { $connect->query("ALTER TABLE orders ADD COLUMN base_amount INT NULL"); } catch (Throwable $e) {}
    try { $connect->query("ALTER TABLE orders ADD COLUMN discount_code VARCHAR(64) NULL"); } catch (Throwable $e) {}
    try { $connect->query("ALTER TABLE orders ADD COLUMN discount_amount INT NOT NULL DEFAULT 0"); } catch (Throwable $e) {}
}

function get_discount_by_code(mysqli $connect, string $code): ?array {
    $sql = "SELECT * FROM discount_codes WHERE code = ? LIMIT 1";
    $stmt = $connect->prepare($sql);
    $stmt->bind_param('s', $code);
    $stmt->execute();
    $res = $stmt->get_result();
    $row = $res->fetch_assoc();
    $stmt->close();
    return $row ?: null;
}

function can_use_discount_for_user(mysqli $connect, int $discount_id, string $user_id, ?int $per_user_limit): bool {
    if ($per_user_limit === null) return true;
    $stmt = $connect->prepare("SELECT COUNT(*) AS cnt FROM discount_usages WHERE discount_id=? AND user_id=? AND is_confirmed=1");
    $stmt->bind_param('is', $discount_id, $user_id);
    $stmt->execute();
    $cnt = ($stmt->get_result()->fetch_assoc()['cnt'] ?? 0);
    $stmt->close();
    return ($cnt < $per_user_limit);
}

function apply_discount_code(mysqli $connect, string $order_code, string $user_id, string $code): array {
    ensure_discount_schema($connect);

    $code = trim($code);
    if ($code === '') return ['success' => false, 'message' => 'کد تخفیف خالی است.'];

    // Fetch order
    $stmt = $connect->prepare("SELECT * FROM orders WHERE order_code=? LIMIT 1");
    $stmt->bind_param('s', $order_code);
    $stmt->execute();
    $order_res = $stmt->get_result();
    $order = $order_res->fetch_assoc();
    $stmt->close();
    if (!$order || (string)$order['user_id'] !== (string)$user_id) {
        return ['success' => false, 'message' => 'سفارش معتبر یافت نشد.'];
    }

    $base_amount = (int)($order['base_amount'] ?? 0);
    if ($base_amount <= 0) $base_amount = (int)$order['amount'];

    // Fetch discount
    $discount = get_discount_by_code($connect, $code);
    if (!$discount || (int)$discount['is_active'] !== 1) {
        return ['success' => false, 'message' => 'کد تخفیف معتبر نیست یا غیرفعال است.'];
    }

    // Check expiry
    if (!empty($discount['expires_at'])) {
        $now = new DateTime('now', new DateTimeZone('Asia/Tehran'));
        $exp = new DateTime($discount['expires_at']);
        if ($now > $exp) {
            return ['success' => false, 'message' => 'مهلت استفاده از کد به پایان رسیده است.'];
        }
    }

    // Check max uses (confirmed only)
    if (!is_null($discount['max_uses'])) {
        if ((int)$discount['used_count'] >= (int)$discount['max_uses']) {
            return ['success' => false, 'message' => 'ظرفیت این کد تخفیف تکمیل شده است.'];
        }
    }

    // Check per-user limit (confirmed only)
    if (!can_use_discount_for_user($connect, (int)$discount['id'], $user_id, $discount['per_user_limit'])) {
        return ['success' => false, 'message' => 'سهمیه استفاده شما از این کد به اتمام رسیده است.'];
    }

    // Check min order amount
    $minOrderRaw = $discount['min_order_total'] ?? $discount['min_order_amount'] ?? null;
    if ($minOrderRaw !== null) {
        $minOrderValue = (float)$minOrderRaw;
        if ($minOrderValue > 0 && $base_amount < $minOrderValue) {
            return ['success' => false, 'message' => 'مبلغ سفارش برای استفاده از این کد کافی نیست.'];
        }
    }

    // Calculate discount amount
    $discount_amount = 0;
    $type = $discount['type'];
    $value = (float)$discount['value'];
    if ($type === 'percent') {
        $value = max(0, min(100, $value));
        $discount_amount = (int)floor($base_amount * ($value / 100.0));
    } else { // fixed
        $discount_amount = (int)round(max(0, $value));
    }
    $discount_amount = max(0, min($discount_amount, $base_amount));
    $final_amount = $base_amount - $discount_amount;

    // Update order amounts
    $stmt2 = $connect->prepare("UPDATE orders SET base_amount=?, discount_code=?, discount_amount=?, amount=? WHERE order_code=?");
    $stmt2->bind_param('isiis', $base_amount, $code, $discount_amount, $final_amount, $order_code);
    $stmt2->execute();
    $stmt2->close();

    // Reserve usage (not confirmed yet)
    $stmt3 = $connect->prepare("INSERT IGNORE INTO discount_usages (discount_id, order_code, user_id, is_confirmed) VALUES (?, ?, ?, 0)");
    $did = (int)$discount['id'];
    $stmt3->bind_param('iss', $did, $order_code, $user_id);
    $stmt3->execute();
    $stmt3->close();

    return [
        'success' => true,
        'message' => 'کد تخفیف با موفقیت اعمال شد.',
        'final_amount' => $final_amount,
        'discount_amount' => $discount_amount,
        'base_amount' => $base_amount,
        'code' => $code,
    ];
}

function confirm_discount_usage_for_order(mysqli $connect, string $order_code): void {
    // Find usage and mark confirmed; bump used_count atomically
    $stmt = $connect->prepare("SELECT du.discount_id FROM discount_usages du JOIN orders o ON o.order_code = du.order_code WHERE du.order_code=? AND du.is_confirmed=0 LIMIT 1");
    $stmt->bind_param('s', $order_code);
    $stmt->execute();
    $res = $stmt->get_result();
    $row = $res->fetch_assoc();
    $stmt->close();
    if (!$row) return;
    $discount_id = (int)$row['discount_id'];

    $connect->begin_transaction();
    try {
        $stmt1 = $connect->prepare("UPDATE discount_usages SET is_confirmed=1 WHERE discount_id=? AND order_code=?");
        $stmt1->bind_param('is', $discount_id, $order_code);
        $stmt1->execute();
        $stmt1->close();

        $stmt2 = $connect->prepare("UPDATE discount_codes SET used_count = used_count + 1 WHERE id=?");
        $stmt2->bind_param('i', $discount_id);
        $stmt2->execute();
        $stmt2->close();

        $connect->commit();
    } catch (Throwable $e) {
        $connect->rollback();
    }
}

function release_discount_usage_for_order(mysqli $connect, string $order_code): void {
    // If order rejected/cancelled, drop the reservation so users can reuse the code
    $stmt = $connect->prepare("DELETE FROM discount_usages WHERE order_code=? AND is_confirmed=0");
    $stmt->bind_param('s', $order_code);
    $stmt->execute();
    $stmt->close();
}
