<?php
require_once __DIR__ . '/../config.php';

function sanaei_cookie_storage_dir(): string {
    $dir = __DIR__ . '/../data/sanaei_cookies';
    if (!is_dir($dir)) {
        @mkdir($dir, 0777, true);
    }
    return $dir;
}

function sanaei_cookie_path_for_link(?string $link): string {
    $hash = $link ? md5($link) : 'default';
    return sanaei_cookie_storage_dir() . '/cookie_' . $hash . '.txt';
}

function sanaei_resolve_cookie_path(?string $cookiePath, ?string $link = null): string {
    if (!empty($cookiePath)) {
        $dir = dirname($cookiePath);
        if (!is_dir($dir)) {
            @mkdir($dir, 0777, true);
        }
        return $cookiePath;
    }
    return sanaei_cookie_path_for_link($link);
}


function request($url, $method = 'GET', $postFields = null, $cookieFile = null, $headers = []) {
    $curl = curl_init();

    curl_setopt_array($curl, array(
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_ENCODING => '',
        CURLOPT_MAXREDIRS => 10,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_CUSTOMREQUEST => strtoupper($method),
        CURLOPT_SSL_VERIFYPEER => !DISABLE_SSL_VERIFY,  // 🔥 این دو خط
        CURLOPT_SSL_VERIFYHOST => (DISABLE_SSL_VERIFY ? 0 : 2),  // 🔥 رو اضافه کن
    ));

    if (!empty($postFields)) {
        curl_setopt($curl, CURLOPT_POSTFIELDS, $postFields);
    }

    if ($cookieFile) {
        curl_setopt($curl, CURLOPT_COOKIEJAR, $cookieFile);
        curl_setopt($curl, CURLOPT_COOKIEFILE, $cookieFile);
    }

    if (!empty($headers)) {
        curl_setopt($curl, CURLOPT_HTTPHEADER, $headers);
    }

    $response = curl_exec($curl);

    if (curl_errno($curl)) {
        $error = curl_error($curl);
        curl_close($curl);
        return "cURL Error: " . $error;
    }

    curl_close($curl);
    return $response;
}


function login($link, $username, $password, $cookiePath = null) {
    $cookiePath = sanaei_resolve_cookie_path($cookiePath, $link);
    $url = rtrim($link, '/') . "/login";
    $postData = http_build_query([
        'username' => $username,
        'password' => $password
    ]);

    $headers = [
        'Content-Type: application/x-www-form-urlencoded',
        'Accept: application/json'
    ];

    $response = request($url, 'POST', $postData, $cookiePath, $headers);
    $decoded = json_decode($response, true);
    $success = false;
    if (is_array($decoded)) {
        $success = !empty($decoded['success']) || (isset($decoded['msg']) && strtolower((string)$decoded['msg']) === 'success');
    } elseif (is_string($response)) {
        $success = stripos($response, 'success') !== false;
    }

    return [
        'success' => $success,
        'response' => $response,
        'data' => $decoded
    ];
}


function generateUUID() {
    return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
        random_int(0, 0xffff), random_int(0, 0xffff), random_int(0, 0xffff),
        random_int(0, 0x0fff) | 0x4000,
        random_int(0, 0x3fff) | 0x8000,
        random_int(0, 0xffff), random_int(0, 0xffff), random_int(0, 0xffff)
    );
}

function randomString($length = 10) {
    return substr(str_shuffle('abcdefghijklmnopqrstuvwxyz0123456789'), 0, $length);
}

function gbToBytes($gb) {
    return $gb * 1024 * 1024 * 1024;
}

function daysToTimestamp($days) {
    return (time() + ($days * 86400)) * 1000; // به میلی‌ثانیه
}


function addClientToInbound(
    int $totalGB,
    int $days,
    string $url,
    int $inboundId = 1,
    ?string $cookiePath = null
): string {
    $cookiePath = sanaei_resolve_cookie_path($cookiePath, $url);
    $uuid = generateUUID();
    $email = randomString(15);
    $totalBytes = $totalGB * 1024 * 1024 * 1024;
    $expiryTime = daysToTimestamp($days);
    $endpoint = rtrim($url, '/') . "/panel/api/inbounds/addClient";
    $settings = json_encode([
        "clients" => [
            [
                "id" => $uuid,
                "alterId" => 0,
                "email" => $email,
                "limitIp" => 5,
                "totalGB" => $totalBytes,
                "expiryTime" => $expiryTime,
                "enable" => true,
                "tgId" => "",
                "subId" => ""
            ]
        ]
    ]);

    $postData = json_encode([
        "id" => $inboundId,
        "settings" => $settings
    ]);

    $headers = [
        'Accept: application/json',
        'Content-Type: application/json'
    ];

    $decodedResponse = json_decode(request($endpoint, 'POST', $postData, $cookiePath, $headers), true);
        $output = [
        "remark" => $email,
        "uuid"=>$uuid,
        "response" => $decodedResponse
    ];

    // خروجی به صورت JSON
    header('Content-Type: application/json');
    return json_encode($output, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
}
function UpdateClient($uuid,$name,
    int $totalGB,
    int $days,
    string $url,
    int $inboundId = 1,
    ?string $cookiePath = null
): string {
    $cookiePath = sanaei_resolve_cookie_path($cookiePath, $url);
    $totalBytes = $totalGB * 1024 * 1024 * 1024;
    $expiryInput = $days;
    if (is_numeric($expiryInput)) {
        $numeric = (float)$expiryInput;
        if ($numeric <= 3650) {
            $expiryTime = daysToTimestamp($numeric);
        } elseif ($numeric < 1000000000000) {
            // Treat as seconds-based Unix timestamp.
            $expiryTime = (int)round($numeric * 1000);
        } else {
            // Already provided in milliseconds.
            $expiryTime = (int)round($numeric);
        }
    } else {
        $expiryTime = daysToTimestamp(0);
    }
    $endpoint = rtrim($url, '/') . "/panel/api/inbounds/updateClient/" . rawurlencode($uuid);
    $settings = json_encode([
        "clients" => [
            [
                "id" => $uuid,
                "alterId" => 0,
                "email" => $name,
                "limitIp" => 5,
                "totalGB" => $totalBytes,
                "expiryTime" => $expiryTime,
                "enable" => true,
                "tgId" => "",
                "subId" => ""
            ]
        ]
    ]);

    $postData = json_encode([
        "id" => $inboundId,
        "settings" => $settings
    ]);

    $headers = [
        'Accept: application/json',
        'Content-Type: application/json'
    ];

    $decodedResponse = json_decode(request($endpoint, 'POST', $postData, $cookiePath, $headers), true);
    $output = [
        "remark" => $name,
        "uuid" => $uuid,
        'exp' => $expiryTime,
        "response" => $decodedResponse
    ];

    // خروجی به صورت JSON
    header('Content-Type: application/json');
    return json_encode($output, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
}
function ResetClientTraffic(string $email, int $inboundId, string $url, ?string $cookiePath = null): string
{
    $cookiePath = sanaei_resolve_cookie_path($cookiePath, $url);
    // ساخت URL کامل برای درخواست
    $endpoint = rtrim($url, '/') . "/panel/api/inbounds/{$inboundId}/resetClientTraffic/" . rawurlencode($email);

    // هدرها
    $headers = [
        'Accept: application/json',
        'Content-Type: application/json'
    ];

    // ارسال درخواست با استفاده از تابع کمکی request (فرض بر این است که این تابع مثل پروژه قبلی‌تان وجود دارد)
    $response = request($endpoint, 'POST', null, $cookiePath, $headers);
    $decodedResponse = json_decode($response, true);

    // آماده‌سازی خروجی
    $output = [
        "email" => $email,
        "inboundId" => $inboundId,
        "response" => $decodedResponse
    ];

    // خروجی به صورت JSON
    header('Content-Type: application/json');
    return json_encode($output, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
}
function DeleteClient(string $uuid, int $inboundId, string $url, ?string $cookiePath = null): string
{
    $cookiePath = sanaei_resolve_cookie_path($cookiePath, $url);
    // ساخت URL نهایی برای درخواست حذف
    $endpoint = rtrim($url, '/') . "/panel/api/inbounds/{$inboundId}/delClient/" . rawurlencode($uuid);

    // هدرها
    $headers = [
        'Accept: application/json',
        'Content-Type: application/json'
    ];

    // ارسال درخواست POST با استفاده از تابع request
    $response = request($endpoint, 'POST', null, $cookiePath, $headers);
    $decodedResponse = json_decode($response, true);

    // خروجی آماده
    $output = [
        "uuid" => $uuid,
        "inboundId" => $inboundId,
        "response" => $decodedResponse
    ];

    // خروجی JSON با فرمت خوانا
    header('Content-Type: application/json');
    return json_encode($output, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
}

function getEmailStats($email, $data) {
    // بررسی می‌کنیم که آیا 'clientStats' داخل داده موجود است
    if (isset($data['clientStats']) && is_array($data['clientStats'])) {
        // Loop through each email in the clientStats
        foreach ($data['clientStats'] as $client) {
            // اگر ایمیل با ایمیل مورد نظر تطابق داشت
            if ($client['email'] === $email) {
                // برگرداندن اطلاعات مورد نظر در قالب یک آرایه
                return [
                    'id' => $client['id'],
                    'inboundId' => $client['inboundId'],
                    'enable' => $client['enable'],
                    'email' => $client['email'],
                    'up' => $client['up'],
                    'down' => $client['down'],
                    'expiryTime' => $client['expiryTime'],
                    'total' => $client['total'],
                    'reset' => $client['reset']
                ];
            }
        }
    }

    // در صورتی که ایمیل پیدا نشد
    return null;
}
function getClientTraffic($link,$email, $cookiePath = null) {
    $cookiePath = sanaei_resolve_cookie_path($cookiePath, $link);
    // ساخت URL بر اساس ایمیل
    $url = rtrim($link, '/') . "/panel/api/inbounds/getClientTraffics/" . rawurlencode($email);

    // هدرهایی که باید به درخواست اضافه شوند
    $headers = [
        'Accept: application/json'
    ];

    // ارسال درخواست با استفاده از تابع request (مشابه با درخواست‌های قبلی)
    $response = request($url, 'GET', null, $cookiePath, $headers);

    // بازگشت پاسخ دریافت شده
    return $response;
}

function getClientTrafficById($link, $uuid, $cookiePath = null) {
    $cookiePath = sanaei_resolve_cookie_path($cookiePath, $link);
    $url = rtrim($link, '/') . "/panel/api/inbounds/getClientTrafficsById/" . rawurlencode($uuid);

    $headers = [
        'Accept: application/json'
    ];

    return request($url, 'GET', null, $cookiePath, $headers);
}
function getPortById($id,  $baseUrl, $cookiePath = null) {
    $cookiePath = sanaei_resolve_cookie_path($cookiePath, $baseUrl);
    $url = rtrim($baseUrl, '/') . '/panel/api/inbounds/list';

    $headers = [
        'Accept: application/json'
    ];

    $response = request($url, 'GET', null, $cookiePath, $headers);
    $data = json_decode($response, true);

    if (!isset($data['obj']) || !is_array($data['obj'])) {
        return json_encode([
            'status' => false,
            'msg' => 'Invalid response from server',
            'status_code' => 500
        ]);
    }

    foreach ($data['obj'] as $value) {
        if ($value['id'] == $id) {
            return json_encode([
                'status' => true,
                'port' => $value['port'],
                'status_code' => 200
            ]);
        }
    }

    return json_encode([
        'status' => false,
        'msg' => 'ID not found',
        'status_code' => 404
    ]);
}

function sanaei_normalize_bool($value): ?bool {
    if (is_bool($value)) {
        return $value;
    }
    if (is_numeric($value)) {
        return ((int)$value) === 1;
    }
    if (is_string($value)) {
        $value = strtolower(trim($value));
        if (in_array($value, ['true', '1', 'yes', 'enable', 'enabled'], true)) {
            return true;
        }
        if (in_array($value, ['false', '0', 'no', 'disable', 'disabled'], true)) {
            return false;
        }
    }
    return null;
}

function sanaei_extract_client_entry($obj, string $identifier) {
    if (!is_array($obj)) {
        return null;
    }

    if (isset($obj['up']) || isset($obj['down']) || isset($obj['email'])) {
        return $obj;
    }

    if (isset($obj['clientStats']) && is_array($obj['clientStats'])) {
        foreach ($obj['clientStats'] as $entry) {
            if (!is_array($entry)) {
                continue;
            }
            $candidate = $entry['email'] ?? $entry['remark'] ?? ($entry['id'] ?? null);
            if ($candidate !== null && $candidate == $identifier) {
                return $entry;
            }
        }
        foreach ($obj['clientStats'] as $entry) {
            if (is_array($entry)) {
                return $entry;
            }
        }
    }

    foreach ($obj as $entry) {
        if (!is_array($entry)) {
            continue;
        }
        $candidate = $entry['email'] ?? $entry['remark'] ?? ($entry['id'] ?? null);
        if ($candidate !== null && $candidate == $identifier) {
            return $entry;
        }
    }

    foreach ($obj as $entry) {
        if (is_array($entry)) {
            return $entry;
        }
    }

    return null;
}

function sanaei_process_usage_response($response, string $identifier): array {
    if (!is_string($response) || $response === '') {
        return ['success' => false, 'error' => 'empty_response'];
    }

    $decoded = json_decode($response, true);
    if (!is_array($decoded)) {
        return ['success' => false, 'error' => 'invalid_json', 'raw' => $response];
    }

    if (isset($decoded['success']) && !$decoded['success']) {
        $message = $decoded['msg'] ?? $decoded['message'] ?? 'request_failed';
        return ['success' => false, 'error' => $message, 'response' => $decoded];
    }

    $obj = $decoded['obj'] ?? null;
    if ($obj === null) {
        return ['success' => false, 'error' => 'missing_obj', 'response' => $decoded];
    }

    $entry = sanaei_extract_client_entry($obj, $identifier);
    if (!$entry || !is_array($entry)) {
        return ['success' => false, 'error' => 'client_not_found', 'response' => $decoded];
    }

    $up = (float)($entry['up'] ?? 0);
    $down = (float)($entry['down'] ?? 0);
    $used = $up + $down;
    $total = isset($entry['total']) ? (float)$entry['total'] : null;
    $expiry = $entry['expiryTime'] ?? ($entry['exp'] ?? null);
    $enableRaw = $entry['enable'] ?? ($entry['status'] ?? null);
    $enable = sanaei_normalize_bool($enableRaw);
    if ($enable === null) {
        $enable = (bool)$enableRaw;
    }

    return [
        'success' => true,
        'data' => [
            'up' => $up,
            'down' => $down,
            'used' => $used,
            'total' => $total,
            'expiryTime' => $expiry,
            'enable' => $enable,
            'raw' => $entry
        ]
    ];
}

function sanaei_fetch_client_usage(string $link, string $identifier, ?string $cookiePath = null, ?string $uuid = null): array {
    $cookiePath = sanaei_resolve_cookie_path($cookiePath, $link);
    $response = getClientTraffic($link, $identifier, $cookiePath);
    $result = sanaei_process_usage_response($response, $identifier);

    if ($result['success'] || empty($uuid)) {
        return $result;
    }

    $fallbackResponse = getClientTrafficById($link, $uuid, $cookiePath);
    $fallbackResult = sanaei_process_usage_response($fallbackResponse, $uuid);
    if ($fallbackResult['success']) {
        $fallbackResult['lookup_method'] = 'uuid';
        return $fallbackResult;
    }

    $result['fallback'] = $fallbackResult;
    return $result;
}

