<?php
require_once __DIR__ . '/includes/auth.php';
require_admin_login();
?>
<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>پنل مدیریت</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Vazirmatn:wght@400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <script>
        if (localStorage.getItem('theme') === 'light') {
            document.documentElement.classList.add('light');
        }
    </script>
    <link rel="stylesheet" href="assets/styles.css?v=liquid_final_fix">
</head>
<body class="glass">

    <div class="flex">
        <?php include __DIR__ . '/partials/sidebar.php'; ?>

        <main id="main-content" class="content mr-0 md:mr-64 p-4 md:p-8 w-full">
            <!-- Mobile top bar -->
            <div class="md:hidden mb-4 flex items-center justify-between">
                <button id="mobile-menu-btn" class="inline-flex items-center gap-2 bg-slate-700/50 text-white px-4 py-2 rounded-lg shadow hover:bg-slate-600/50 transition">
                    <i class="fas fa-bars"></i>
                    <span>منو</span>
                </button>
            </div>
            
            <section id="dashboard" class="page-content">
                <h2 class="text-2xl md:text-3xl font-bold mb-6 text-slate-100">داشبورد</h2>
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6" id="stats-grid">
                    <!-- Stat cards will be loaded here by JS -->
                </div>
                <div class="mt-6 grid grid-cols-1 xl:grid-cols-4 gap-4" id="financial-overview"></div>
                <div class="mt-6 grid grid-cols-1 xl:grid-cols-2 gap-6" id="dashboard-insights">
                    <div class="bg-white p-6" id="revenue-trend-card">
                        <div class="flex items-center justify-between mb-4">
                            <h3 class="widget-title font-medium">روند فروش ۱۴ روز اخیر</h3>
                            <span class="text-xs text-slate-400">بروزشده خودکار</span>
                        </div>
                        <div id="revenue-trend" class="min-h-[180px]"></div>
                    </div>
                    <div class="bg-white p-6" id="order-timeline-card">
                        <div class="flex items-center justify-between mb-4">
                            <h3 class="widget-title font-medium">آخرین سفارش‌ها</h3>
                            <button id="refresh-orders-timeline" class="liquid-pill px-3 py-1 text-xs flex items-center gap-2"><i class="fas fa-rotate"></i><span>بروزرسانی</span></button>
                        </div>
                        <div id="order-timeline"></div>
                    </div>
                </div>
                <!-- Enhanced widgets will be loaded here by JS -->
            </section>

                        <section id="users-list" class="page-content hidden">
                <h2 class="text-2xl md:text-3xl font-bold mb-6 text-slate-100">لیست کاربران</h2>
                <div class="bg-white p-6 rounded-xl space-y-4">
                    <div class="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
                        <div class="relative md:flex-1">
                            <i class="fas fa-search absolute right-4 top-1/2 -translate-y-1/2 text-gray-400"></i>
                            <input type="text" id="users-list-search" placeholder="جستجو در کاربران..." class="w-full p-3 pr-10 rounded-xl">
                        </div>
                        <div class="flex items-center gap-3">
                            <button id="users-export-btn" class="liquid-pill px-4 py-2 text-sm flex items-center gap-2"><i class="fas fa-file-arrow-down"></i><span>خروجی CSV</span></button>
                        </div>
                    </div>
                    <div class="grid grid-cols-1 md:grid-cols-3 gap-4 mb-4">
                        <select id="users-list-status-filter" class="w-full p-3 rounded-xl">
                            <option value="">همه وضعیت‌ها</option>
                            <option value="active">دارای سرویس فعال</option>
                            <option value="inactive">بدون سرویس فعال</option>
                            <option value="none">بدون سرویس</option>
                        </select>
                        <select id="users-list-server-filter" class="w-full p-3 rounded-xl">
                            <option value="">همه سرورها</option>
                        </select>
                        <div class="hidden md:block"></div>
                    </div>
                    <div id="users-table-container" class="overflow-x-auto"></div>
                    <div id="users-pagination" class="mt-4 flex justify-center"></div>
                </div>
            </section>

            <section id="services-list" class="page-content hidden">
                 <h2 class="text-2xl md:text-3xl font-bold mb-6 text-slate-100">لیست سرویس‌ها</h2>
                <div class="bg-white p-6">
                    <div class="grid grid-cols-1 md:grid-cols-3 gap-4 mb-4">
                        <input type="text" id="services-list-search" placeholder="جستجو (نام، UUID, ID کاربر)..." class="w-full p-3">
                        <select id="services-list-server-filter" class="w-full p-3"></select>
                        <select id="services-list-status-filter" class="w-full p-3">
                            <option value="">همه وضعیت‌ها</option>
                            <option value="active">فعال</option>
                            <option value="inactive">غیرفعال</option>
                        </select>
                    </div>
                    <div id="services-table-container" class="overflow-x-auto p-2"></div>
                    <div id="services-pagination" class="mt-4 flex justify-center"></div>
                </div>
            </section>

            <section id="user-search" class="page-content hidden">
                <h2 class="text-2xl md:text-3xl font-bold mb-6 text-slate-100">جستجوی کاربر</h2>
                <div class="bg-white p-6">
                    <div class="mb-4 relative">
                        <i class="fas fa-search absolute right-4 top-1/2 -translate-y-1/2 text-gray-400"></i>
                        <input type="text" id="user-search-input" placeholder="جستجوی کاربر با ID، یوزرنیم یا UUID..." class="w-full p-3 pr-10">
                    </div>
                    <div id="user-details-container" class="mt-4">
                        <p class="text-center py-8">برای مشاهده اطلاعات، کاربر را جستجو کنید.</p>
                    </div>
                </div>
            </section>

            <section id="servers" class="page-content hidden">
                 <h2 class="text-2xl md:text-3xl font-bold mb-6 text-slate-100">مدیریت سرورها</h2>
                 <div class="bg-white p-6">
                    <button id="add-server-btn" class="bg-gradient-to-r from-green-500 to-emerald-600 text-white px-5 py-2.5 rounded-lg shadow-md hover:shadow-lg mb-4 font-semibold transition-all duration-300"><i class="fas fa-plus ml-2"></i>افزودن سرور جدید</button>
                    <div id="server-list" class="overflow-x-auto p-2"></div>
                 </div>
                 <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mt-6">
                    <div class="bg-white p-6" id="server-health-card">
                        <div class="flex items-center justify-between mb-4">
                            <h3 class="widget-title font-medium">نمای کلی سلامت سرورها</h3>
                            <span class="text-xs text-slate-400">به‌روز رسانی هنگام باز کردن صفحه</span>
                        </div>
                        <div id="server-health-overview" class="space-y-4"></div>
                    </div>
                    <div class="bg-white p-6" id="server-tags-card">
                        <div class="flex items-center justify-between mb-4">
                            <h3 class="widget-title font-medium">برچسب‌های سرور</h3>
                            <button id="refresh-server-tags" class="liquid-pill px-3 py-1 text-xs flex items-center gap-2"><i class="fas fa-rotate"></i><span>بروزرسانی</span></button>
                        </div>
                        <form id="server-tag-form" class="grid grid-cols-1 md:grid-cols-2 gap-3 mb-4">
                            <input type="text" name="name" placeholder="عنوان برچسب" class="p-2" required>
                            <input type="text" name="color" placeholder="#6366f1" class="p-2">
                            <button type="submit" class="md:col-span-2 bg-indigo-600 hover:bg-indigo-700 text-white px-4 py-2 rounded-lg transition">افزودن برچسب</button>
                        </form>
                        <div id="server-tags-list" class="space-y-3"></div>
                    </div>
                 </div>
            </section>
            
            <section id="plans" class="page-content hidden">
                 <h2 class="text-2xl md:text-3xl font-bold mb-6 text-slate-100">مدیریت تعرفه‌ها</h2>
                 <div class="bg-white p-6">
                    <button id="add-plan-btn" class="bg-gradient-to-r from-green-500 to-emerald-600 text-white px-5 py-2.5 rounded-lg shadow-md hover:shadow-lg mb-4 font-semibold transition-all duration-300"><i class="fas fa-plus ml-2"></i>افزودن تعرفه جدید</button>
                    <div id="plan-list" class="overflow-x-auto p-2"></div>
                </div>
            </section>

            <section id="pricing" class="page-content hidden">
                <h2 class="text-2xl md:text-3xl font-bold mb-6 text-slate-100">مدیریت قیمت‌ها</h2>
                <div class="bg-white p-6">
                    <div class="mb-4">
                        <label for="pricing-server-select" class="block mb-2 font-semibold">یک سرور را برای مدیریت قیمت‌ها انتخاب کنید:</label>
                        <select id="pricing-server-select" class="w-full p-3"></select>
                    </div>
                    <div id="price-list" class="mt-4"></div>
                </div>
            </section>

            <section id="discounts" class="page-content hidden">
                <h2 class="text-2xl md:text-3xl font-bold mb-6 text-slate-100">کدهای تخفیف</h2>
                <div class="bg-white p-6">
                    <div class="flex justify-between items-center mb-4">
                        <div class="text-sm">کدهای فعال و تاریخ‌دار</div>
                        <button id="add-discount-btn" class="bg-indigo-600 hover:bg-indigo-700 text-white px-4 py-2 rounded-lg transition">
                            افزودن کد
                        </button>
                    </div>
                    <div id="discounts-list" class="overflow-x-auto p-2"></div>
                </div>
            </section>

            <section id="messages" class="page-content hidden">
                <div class="flex justify-between items-center mb-6">
                    <h2 class="text-2xl md:text-3xl font-bold text-slate-100">مدیریت پیام‌ها</h2>
                    <button id="help-variables-btn" class="bg-gradient-to-r from-sky-500 to-indigo-500 text-white px-5 py-2.5 rounded-lg shadow-md hover:shadow-lg font-semibold transition-all duration-300"><i class="fas fa-question-circle ml-2"></i>راهنمای متغیرها</button>
                </div>
                <div class="bg-white p-6">
                    <div class="mb-4"><input type="text" id="messages-search" placeholder="جستجو در کلید، متن یا توضیحات پیام..." class="w-full p-3"></div>
                    <div id="messages-list" class="overflow-x-auto"></div>
                </div>
            </section>

            <section id="broadcast" class="page-content hidden">
                <h2 class="text-2xl md:text-3xl font-bold mb-6 text-slate-100">پیام همگانی</h2>
                <div class="bg-white p-6">
                    <textarea id="broadcast-message" class="w-full p-3 mb-4" rows="6" placeholder="پیام خود را اینجا بنویسید..."></textarea>
                    <button id="send-broadcast-btn" class="w-full bg-gradient-to-r from-indigo-500 to-purple-600 text-white px-4 py-3 rounded-lg shadow-md hover:shadow-lg font-semibold transition-all duration-300">ارسال پیام به همه کاربران</button>
                     <div id="broadcast-status" class="mt-4 text-center"></div>
                </div>
            </section>

            <section id="settings" class="page-content hidden">
                <div class="bg-white p-4 md:p-5 rounded-xl border border-white/10 mb-4 flex items-center justify-between sticky top-2">
                    <h2 class="text-2xl md:text-3xl font-bold text-slate-100">تنظیمات</h2>
                    <button id="save-settings-btn" class="btn-primary btn-sm">ذخیره تنظیمات</button>
                </div>
                <div class="bg-white p-6 space-y-8" id="settings-cards">
                    
                    <div>
                        <label for="admin-id-input" class="block mb-2 font-semibold">شناسه عددی ادمین اصلی</label>
                        <input type="number" id="admin-id-input" class="w-full p-3" placeholder="مثال: 123456789">
                        <p class="text-xs text-gray-400 mt-1">این شناسه برای دریافت گزارش‌ها و مدیریت ربات استفاده می‌شود.</p>
                    </div>

                    <!-- Payments Section wrapper start -->
                    <div class="rounded-xl bg-white/5 border border-white/10 p-4 md:p-5">
                      <div class="flex items-center gap-3 mb-4">
                        <i class="fas fa-credit-card text-emerald-300"></i>
                        <h3 class="font-bold">پرداخت‌ها</h3>
                      </div>
                    
                    <div class="border-t border-slate-600 pt-6">
                        <label class="flex items-center cursor-pointer">
                            <input type="checkbox" id="card-payment-status-toggle" class="hidden peer">
                            <span class="toggle-track w-12 h-6 rounded-full bg-gray-600 peer-checked:bg-green-500 transition-all duration-300 relative"></span>
                            <span class="mr-3 font-medium">پرداخت کارت به کارت</span>
                        </label>
                         <div id="card-payment-details" class="mt-4 hidden">
                            <label for="card-number-input" class="block mb-2 font-semibold">اطلاعات کارت</label>
                            <textarea id="card-number-input" class="w-full p-3" rows="3" placeholder="مثال: 6219860000000000 - نام دارنده"></textarea>
                        </div>
                    </div>

                    <div class="border-t border-slate-600 pt-6">
                        <label class="flex items-center cursor-pointer">
                            <input type="checkbox" id="crypto-payment-status-toggle" class="hidden peer">
                            <span class="toggle-track w-12 h-6 rounded-full bg-gray-600 peer-checked:bg-green-500 transition-all duration-300 relative"></span>
                            <span class="mr-3 font-medium">پرداخت با رمز ارز</span>
                        </label>
                         <div id="crypto-payment-details" class="mt-4 hidden">
                            <label for="crypto-wallets-input" class="block mb-2 font-semibold">آدرس کیف پول‌های رمز ارز</label>
                            <textarea id="crypto-wallets-input" class="w-full p-3" rows="4" placeholder="هر آدرس در یک خط جداگانه وارد شود.
مثال:
TRC20 (USDT): T...
BTC: 1..."></textarea>
                        </div>
                    </div>
                    <div class="border-t border-slate-600 pt-6">
                        <label class="flex items-center cursor-pointer">
                            <input type="checkbox" id="stars-payment-status-toggle" class="hidden peer">
                            <span class="toggle-track w-12 h-6 rounded-full bg-gray-600 peer-checked:bg-green-500 transition-all duration-300 relative"></span>
                            <span class="mr-3 font-medium">پرداخت با استارز</span>
                        </label>
                    </div>
                    <div class="border-t border-slate-600 pt-6">
                        <label class="flex items-center cursor-pointer">
                            <input type="checkbox" id="kyc-status-toggle" class="hidden peer">
                            <span class="toggle-track w-12 h-6 rounded-full bg-gray-600 peer-checked:bg-green-500 transition-all duration-300 relative"></span>
                            <span class="mr-3 font-medium">احراز هویت (برای کارت به کارت)</span>
                        </label>
                         <div id="kyc-details" class="mt-4 hidden">
                            <div class="mb-4">
                              <label for="kyc-channel-id-input" class="block mb-2 font-semibold">آیدی کانال احراز هویت</label>
                              <input type="text" id="kyc-channel-id-input" class="w-full p-2" placeholder="مثال: -100123456789">
                            </div>
                            <div>
                              <label for="kyc-message-input" class="block mb-2 font-semibold">پیام راهنمای احراز هویت</label>
                              <textarea id="kyc-message-input" class="w-full p-3" rows="4"></textarea>
                            </div>
                        </div>
                    </div>
                    </div>
                    <!-- Payments Section wrapper end -->
                    <div class="rounded-xl bg-white/5 border border-white/10 p-4 md:p-5 mt-6">
                      <div class="flex items-center gap-3 mb-4">
                        <i class="fas fa-user-check text-sky-300"></i>
                        <h3 class="font-bold">عضویت و وضعیت ربات</h3>
                      </div>
                      <div class="border-t border-slate-600 pt-6">
                        <label class="flex items-center cursor-pointer">
                            <input type="checkbox" id="join-status-toggle" class="hidden peer">
                            <span class="toggle-track w-12 h-6 rounded-full bg-gray-600 peer-checked:bg-green-500 transition-all duration-300 relative"></span>
                            <span class="mr-3 font-medium">عضویت اجباری در کانال</span>
                        </label>
                         <div class="mt-4">
                            <label for="join-channels-input" class="block mb-2 font-semibold">کانال‌های عضویت اجباری</label>
                            <textarea id="join-channels-input" class="w-full p-3" rows="3" placeholder="هر یوزرنیم کانال در یک خط (با @)"></textarea>
                            <p class="text-xs text-gray-400 mt-1">مثال:
@channel1
@channel2</p>
                        </div>
                      </div>
                      
                      <div class="border-t border-slate-600 pt-6">
                         <label class="flex items-center cursor-pointer">
                            <input type="checkbox" id="bot-status-toggle" class="hidden peer">
                            <span class="toggle-track w-12 h-6 rounded-full bg-gray-600 peer-checked:bg-green-500 transition-all duration-300 relative"></span>
                            <span class="mr-3 font-medium">فعال بودن ربات (خاموش کردن = حالت تعمیرات)</span>
                        </label>
                      </div>
                    </div>
                    <div class="rounded-xl bg-white/5 border border-white/10 p-4 md:p-5 mt-6">
                      <div class="flex items-center justify-between mb-4">
                        <div class="flex items-center gap-3">
                          <i class="fas fa-gift text-pink-300"></i>
                          <h3 class="font-bold">برنامه دعوت از دوستان</h3>
                        </div>
                        <label class="flex items-center cursor-pointer">
                            <input type="checkbox" id="referral-status-toggle" class="hidden peer">
                            <span class="toggle-track w-12 h-6 rounded-full bg-gray-600 peer-checked:bg-green-500 transition-all duration-300 relative"></span>
                            <span class="mr-3 font-medium">فعال سازی</span>
                        </label>
                      </div>
                      <p class="text-xs text-slate-400 mb-4">پاداش و شرایط برنامه دعوت را شخصی‌سازی کنید. با فعال بودن این بخش، کاربران پس از اولین خرید دوستانشان کد تخفیف دریافت می‌کنند.</p>
                      <div id="referral-settings-body" class="grid grid-cols-1 md:grid-cols-2 gap-4 hidden">
                        <div>
                          <label for="referral-type-select" class="block mb-2 font-semibold">نوع پاداش</label>
                          <select id="referral-type-select" class="w-full p-3">
                            <option value="fixed">مبلغ ثابت</option>
                            <option value="percent">درصدی</option>
                          </select>
                        </div>
                        <div>
                          <label for="referral-value-input" class="block mb-2 font-semibold">مقدار پاداش</label>
                          <input type="number" id="referral-value-input" class="w-full p-3" min="0" step="1" placeholder="مثال: 50000">
                          <p class="text-xs text-slate-400 mt-1">برای پاداش درصدی عددی بین 1 تا 100 وارد کنید.</p>
                        </div>
                        <div>
                          <label for="referral-min-order-input" class="block mb-2 font-semibold">حداقل مبلغ خرید</label>
                          <input type="number" id="referral-min-order-input" class="w-full p-3" min="0" step="1000" placeholder="مثال: 100000">
                        </div>
                        <div>
                          <label for="referral-expire-input" class="block mb-2 font-semibold">انقضای کد (روز)</label>
                          <input type="number" id="referral-expire-input" class="w-full p-3" min="0" step="1" placeholder="مثال: 30">
                        </div>
                        <div>
                          <label for="referral-prefix-input" class="block mb-2 font-semibold">پیشوند کد تخفیف</label>
                          <input type="text" id="referral-prefix-input" class="w-full p-3" maxlength="12" placeholder="مثال: REF">
                        </div>
                        <div>
                          <label for="referral-limit-input" class="block mb-2 font-semibold">تعداد دفعات استفاده</label>
                          <input type="number" id="referral-limit-input" class="w-full p-3" min="1" step="1" placeholder="معمولاً 1">
                        </div>
                      </div>
                    </div>
                    <button id="save-settings-btn-bottom" class="hidden w-full bg-gradient-to-r from-blue-500 to-cyan-500 text-white px-4 py-3 rounded-lg shadow-md hover:shadow-lg font-semibold transition-all duration-300">ذخیره تنظیمات</button>
                </div>
            </section>
        </main>
    </div>
    
    <div id="modal" class="modal fixed inset-0 bg-gray-900 bg-opacity-70 hidden items-center justify-center z-30 p-4 opacity-0 -translate-y-10">
        <div id="modal-content" class="rounded-xl shadow-2xl w-full max-w-2xl p-6 transform transition-all scale-95">
            <!-- Modal content will be loaded here by JS -->
        </div>
    </div>

    <!-- Left slide-out drawer for user details -->
    <div id="user-drawer" class="fixed inset-0 z-40 hidden">
        <div id="user-drawer-overlay" class="absolute inset-0 bg-black/50"></div>
        <aside id="user-drawer-panel" class="absolute left-0 top-0 h-full w-[90%] max-w-md bg-slate-800 text-slate-200 border-r border-white/10 transform -translate-x-full transition-transform duration-300 shadow-2xl">
            <div class="flex items-center justify-between p-4 border-b border-white/10">
                <h3 class="font-bold">جزئیات کاربر</h3>
                <button id="user-drawer-close" class="btn-icon"><i class="fas fa-xmark"></i></button>
            </div>
            <div id="user-drawer-body" class="p-4 overflow-y-auto h-[calc(100%-56px)]"></div>
        </aside>
    </div>

    <script type="text/plain" id="legacy-script">
    document.addEventListener('DOMContentLoaded', function() {
        // =================================================================
        // ===== CORE UI & NAVIGATION =====
        // =================================================================
        
        const sidebarLinks = document.querySelectorAll('.sidebar-link');
        const contentPages = document.querySelectorAll('.page-content');
        const modal = document.getElementById('modal');
        const modalContent = document.getElementById('modal-content');
        const sidebarEl = document.getElementById('sidebar');
        const overlayEl = document.getElementById('sidebar-overlay');
        const mobileBtn = document.getElementById('mobile-menu-btn');
        // Drawer elements
        const userDrawer = document.getElementById('user-drawer');
        const userDrawerPanel = document.getElementById('user-drawer-panel');
        const userDrawerBody = document.getElementById('user-drawer-body');
        const userDrawerOverlay = document.getElementById('user-drawer-overlay');
        const userDrawerClose = document.getElementById('user-drawer-close');

        function openSidebar() { sidebarEl.classList.remove('translate-x-full'); overlayEl.classList.remove('hidden'); }
        function closeSidebar() { sidebarEl.classList.add('translate-x-full'); overlayEl.classList.add('hidden'); }
        mobileBtn?.addEventListener('click', openSidebar);
        overlayEl?.addEventListener('click', closeSidebar);
        window.addEventListener('resize', () => { if (window.innerWidth >= 768) overlayEl?.classList.add('hidden'); });

        // User drawer helpers
        function openUserDrawer() {
            if (!userDrawer || !userDrawerPanel) return;
            userDrawer.classList.remove('hidden');
            setTimeout(() => userDrawerPanel.classList.remove('-translate-x-full'), 10);
        }
        function closeUserDrawer() {
            if (!userDrawer || !userDrawerPanel) return;
            userDrawerPanel.classList.add('-translate-x-full');
            setTimeout(() => userDrawer.classList.add('hidden'), 300);
        userDrawerOverlay?.addEventListener('click', closeUserDrawer);
        userDrawerClose?.addEventListener('click', closeUserDrawer);
        document.addEventListener('keydown', (e) => { if (e.key === 'Escape') closeUserDrawer(); });

        sidebarLinks.forEach(link => {
            link.addEventListener('click', function(e) {
                e.preventDefault();
                sidebarLinks.forEach(l => l.classList.remove('active'));
                this.classList.add('active');
                const targetId = this.getAttribute('data-target');
                contentPages.forEach(page => page.classList.toggle('hidden', page.id !== targetId));
                loadPageContent(targetId);
                if (window.innerWidth < 768) { closeSidebar(); }
            });
        });

        function openModal(htmlContent) {
            modalContent.innerHTML = htmlContent;
            modal.classList.remove('hidden');
            setTimeout(() => {
                modal.classList.remove('opacity-0', '-translate-y-10');
                modalContent.classList.remove('scale-95');
            }, 10);
        window.openModal = openModal;

        function closeModal() {
            modal.classList.add('opacity-0', '-translate-y-10');
            modalContent.classList.add('scale-95');
            setTimeout(() => {
                modal.classList.add('hidden');
                modalContent.innerHTML = '';
            }, 300);
        window.closeModal = closeModal;
        modal.addEventListener('click', (e) => { if (e.target === modal) closeModal(); });

        async function apiRequest(action, params = {}) {
            const urlParams = new URLSearchParams({ action, ...params });
            try {
                const response = await fetch('api.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    credentials: 'same-origin',
                    cache: 'no-store',
                    body: urlParams
                });
                if (!response.ok) throw new Error(`HTTP error! status: ${response.status}`);
                return await response.json();
            } catch (error) {
                console.error('API Request Error:', error);
                openModal(`<div class="text-center"><h3 class="text-xl font-bold text-red-500 mb-4">خطای ارتباط</h3><p>خطا در برقراری ارتباط با سرور.</p><button onclick="closeModal()" class="mt-6 bg-red-500 text-white px-6 py-2 rounded-lg">بستن</button></div>`);
                return { success: false, message: 'خطا در ارتباط' };
        window.apiRequest = apiRequest;
        
        function showLoader(container) {
            container.innerHTML = '<div class="flex justify-center p-8"><div class="loader"></div></div>';
        window.showLoader = showLoader;

        function renderPagination(containerId, currentPage, totalPages, callback) {
            const container = document.getElementById(containerId);
            if (!container) return;
            if (totalPages <= 1) { container.innerHTML = ''; return; }
            
            const pagesToShow = 5;
            let startPage = Math.max(1, currentPage - Math.floor(pagesToShow / 2));
            let endPage = Math.min(totalPages, startPage + pagesToShow - 1);
            if (endPage - startPage + 1 < pagesToShow) startPage = Math.max(1, endPage - pagesToShow + 1);

            let html = '<div class="flex items-center justify-center space-x-2 space-x-reverse">';
            html += `<button data-page="${currentPage - 1}" class="pagination-btn" ${currentPage === 1 ? 'disabled' : ''}><i class="fas fa-chevron-right"></i></button>`;

            if (startPage > 1) {
                html += `<button data-page="1" class="pagination-btn">1</button>`;
                if (startPage > 2) html += `<span class="pagination-ellipsis">...</span>`;

            for (let i = startPage; i <= endPage; i++) {
                html += `<button data-page="${i}" class="pagination-btn ${i === currentPage ? 'active' : ''}">${i}</button>`;

            if (endPage < totalPages) {
                 if (endPage < totalPages - 1) html += `<span class="pagination-ellipsis">...</span>`;
                html += `<button data-page="${totalPages}" class="pagination-btn">${totalPages}</button>`;

            html += `<button data-page="${currentPage + 1}" class="pagination-btn" ${currentPage === totalPages ? 'disabled' : ''}><i class="fas fa-chevron-left"></i></button></div>`;
            
            container.innerHTML = html;
            container.querySelectorAll('.pagination-btn:not([disabled])').forEach(button => {
                button.addEventListener('click', () => callback(parseInt(button.dataset.page)));
            });

        // =================================================================
        // ===== PAGE CONTENT LOADERS =====
        // =================================================================

        function loadPageContent(pageId, params = {}) {
            const pageLoaders = {
                'dashboard': loadDashboard,
                'users-list': () => loadUsersList(params.page || 1, params.search || '', document.getElementById('users-list-status-filter')?.value || '', document.getElementById('users-list-server-filter')?.value || ''),
                'user-search': () => { document.getElementById('user-details-container').innerHTML = '<p class="text-center py-8">برای مشاهده اطلاعات، کاربر را جستجو کنید.</p>'; },
                'servers': loadServers,
                'plans': loadPlans,
                'pricing': loadPricing,
                'discounts': loadDiscounts,
                'messages': loadMessages,
                'broadcast': () => { document.getElementById('broadcast-status').innerHTML = ''; },
                'settings': loadSettings,
            };
            pageLoaders[pageId]?.();
        
        async function loadDashboard() {
            const statsGrid = document.getElementById('stats-grid');
            showLoader(statsGrid);

            const data = await apiRequest('get_stats');
            if (data.success) {
                statsGrid.innerHTML = `
                    <div class="bg-white p-5 md:p-6 flex items-center justify-between stat-kpi">
                        <div class="mr-4">
                            <h3 class="font-medium">کل کاربران</h3>
                            <p class="text-2xl md:text-3xl font-bold">${data.stats.total_users}</p>
                        </div>
                        <div class="icon-pill"><i class="fas fa-users"></i></div>
                    </div>
                    <div class="bg-white p-5 md:p-6 flex items-center justify-between stat-kpi">
                        <div class="mr-4">
                            <h3 class="font-medium">سرویس های فعال</h3>
                            <p class="text-2xl md:text-3xl font-bold">${data.stats.total_services}</p>
                        </div>
                        <div class="icon-pill"><i class="fas fa-rocket"></i></div>
                    </div>
                    <div class="bg-white p-5 md:p-6 flex items-center justify-between stat-kpi">
                        <div class="mr-4">
                            <h3 class="font-medium">تعداد سرورها</h3>
                            <p class="text-2xl md:text-3xl font-bold">${data.stats.total_servers}</p>
                        </div>
                        <div class="icon-pill"><i class="fas fa-server"></i></div>
                    </div>`;
            } else {
                statsGrid.innerHTML = '<div class="col-span-full bg-red-500/30 text-red-300 p-4 rounded-lg">خطا در بارگذاری آمار.</div>';

            const dashboardSection = document.getElementById('dashboard');
            let widgets = document.getElementById('dashboard-widgets');
            if (!widgets) {
                widgets = document.createElement('div');
                widgets.id = 'dashboard-widgets';
                widgets.className = 'mt-6 grid grid-cols-1 lg:grid-cols-3 gap-6';
                dashboardSection.appendChild(widgets);
            widgets.innerHTML = `
                <div class="lg:col-span-2 bg-white p-6">
                    <div class="flex items-center justify-between mb-4">
                        <h3 class="widget-title font-medium">فعالیت های اخیر</h3>
                        <button class="liquid-pill px-3 py-1 text-sm" onclick="document.querySelector('[data-target=users-list]').click()">مشاهده همه</button>
                    </div>
                    <div id="recent-services"></div>
                </div>
                <div>
                    <div class="bg-white p-6">
                        <h3 class="widget-title font-medium mb-4">میانبرها</h3>
                        <div id="quick-actions" class="grid grid-cols-2 gap-3"></div>
                    </div>
                    <div class="bg-white p-6 mt-6">
                        <h3 class="widget-title font-medium mb-4">وضعیت سرورها</h3>
                        <div id="servers-mini"></div>
                    </div>
                </div>`;

            document.getElementById('quick-actions').innerHTML = `
                <button class="quick-action" onclick="document.querySelector('[data-target=user-search]').click()"><i class="fas fa-magnifying-glass ml-2"></i><span>جستجوی کاربر</span></button>
                <button class="quick-action" onclick="document.querySelector('[data-target=users-list]').click()"><i class="fas fa-rectangle-list ml-2"></i><span>کاربران/سرویس‌ها</span></button>
                <button class="quick-action" onclick="document.querySelector('[data-target=servers]').click()"><i class="fas fa-server ml-2"></i><span>مدیریت سرورها</span></button>
                <button class="quick-action" onclick="document.querySelector('[data-target=broadcast]').click()"><i class="fas fa-bullhorn ml-2"></i><span>پیام همگانی</span></button>`;
            
            const recentContainer = document.getElementById('recent-services');
            showLoader(recentContainer);
            const svcData = await apiRequest('get_all_services', { page: 1 });
            if (svcData.success && svcData.services.length > 0) {
                const list = svcData.services.slice(0, 5).map(s => `
                    <li class="glass-row">
                        <div class="title">${s.display || ''}</div>
                        <div class="meta">${s.owner || '-'}</div>
                        <div class="meta">${s.location || '-'}</div>
                        <div class="badge">${s.expire ? new Date(s.expire * 1000).toLocaleDateString('fa-IR') : '-'}</div>
                    </li>`).join('');
                recentContainer.innerHTML = `<ul class="glass-list">${list}</ul>`;
            } else {
                recentContainer.innerHTML = '<div class="p-4 text-center">موردی یافت نشد.</div>';

            const serversMiniContainer = document.getElementById('servers-mini');
            showLoader(serversMiniContainer);
            const serversData = await apiRequest('get_servers');
            if (serversData.success && serversData.servers.length > 0) {
                serversMiniContainer.innerHTML = serversData.servers.map(s => {
                    const on = s.status === 'on';
                    return `<div class="flex items-center justify-between py-2 border-b border-white/10 last:border-0">
                        <div class="flex items-center gap-2">
                            <span class="server-dot ${on ? 'on' : 'off'}"></span>
                            <span>${s.display_name || s.name}</span>
                        </div>
                        <span class="text-xs ${on ? 'text-green-400' : 'text-red-400'}">${on ? 'فعال' : 'خاموش'}</span>
                    </div>`;
                }).join('');
            } else {
                serversMiniContainer.innerHTML = '<div class="p-4 text-center">سروری ثبت نشده.</div>';

        async function loadUsersList(page = 1, search = '', status = '', server = '') {
            const container = document.getElementById('users-table-container');
            showLoader(container);
            const data = await apiRequest('get_all_users', { page, search, status, server });
            if (data.success) {
                container.innerHTML = data.users.length > 0 ? `
                    <table class="w-full text-right table-cool">
                        <thead><tr>
                            <th class="p-3">ID کاربر</th>
                            <th class="p-3">نام کاربری</th>
                            <th class="p-3">موجودی (تومان)</th>
                            <th class="p-3">تعداد سرویس</th>
                        </tr></thead>
                        <tbody>${data.users.map(user => `
                            <tr>
                                <td class="p-4"><code class="user-id-link cursor-pointer text-indigo-300 hover:text-indigo-200" data-user-id="${user.id}">${user.id}</code></td>
                                <td class="p-4">${user.username ? '@' + user.username : '<span>-</span>'}</td>
                                <td class="p-4">${new Intl.NumberFormat().format(user.coin)}</td>
                                <td class="p-4">${user.service_count} <span class="text-xs text-slate-400">(${user.active_service_count ?? 0} فعال)</span></td>
                            </tr>`).join('')}
                        </tbody>
                    </table>` : '<p class="text-center p-4">کاربری یافت نشد.</p>';
                renderPagination('users-pagination', data.current_page, data.total_pages, (newPage) => loadUsersList(newPage, search, status, server));

                // Ensure details panel exists below list (if not using the grid version)
                // Drawer handles user details now; no inline details panel

                // Click handlers to load user services into panel
                container.querySelectorAll('.user-id-link').forEach(el => {
                    el.addEventListener('click', async () => {
                        const uid = el.getAttribute('data-user-id');
                        openUserDrawer();
                        if (userDrawerBody) { showLoader(userDrawerBody); }
                        const d = await apiRequest('get_user_details', { identifier: uid });
                        displayUserDetailsPanel(d);
                    });
                });
            } else {
                container.innerHTML = '<p class="text-red-500 text-center p-4">خطا در بارگذاری کاربران.</p>';
        
        async function loadServicesList(page = 1, search = '', server = '', status = '') {
             const container = document.getElementById('services-table-container');
             showLoader(container);
             const data = await apiRequest('get_all_services', { page, search, server, status });
             if (data.success) {
                container.innerHTML = data.services.length > 0 ? `
                    <table class="w-full text-right text-sm table-cool">
                        <thead>
                            <tr>
                                <th class="p-3">نام سرویس</th>
                                <th class="p-3">مالک (ID)</th>
                                <th class="p-3">سرور</th>
                                <th class="p-3">مصرف / کل (GB)</th>
                                <th class="p-3">انقضا</th>
                                <th class="p-3">وضعیت</th>
                                <th class="p-3">عملیات</th>
                            </tr>
                        </thead>
                        <tbody>
                            ${data.services.map(s => `
                                <tr>
                                    <td class="p-4 font-mono text-xs">${s.display}</td>
                                    <td class="p-4"><code>${s.owner}</code></td>
                                    <td class="p-4">${s.location}</td>
                                    <td class="p-4">
                                        ${s.used_traffic !== null ? `
                                            <div class="flex flex-col">
                                                <span class="font-semibold">${s.used_traffic} / ${s.size}</span>
                                                <span class="text-xs text-slate-400">(${s.remaining_traffic} مانده)</span>
                                            </div>
                                        ` : `
                                            <span>- / ${s.size}</span>
                                        `}
                                    </td>
                                    <td class="p-4">${new Date(s.expire * 1000).toLocaleDateString('fa-IR')}</td>
                                    <td class="p-4">${s.is_active ? '<span class="px-2 py-1 bg-green-500/20 text-green-300 rounded-full text-xs font-medium">فعال</span>' : '<span class="px-2 py-1 bg-red-500/20 text-red-300 rounded-full text-xs font-medium">غیرفعال</span>'}</td>
                                    <td class="p-4"><button class="btn-icon text-red-400 hover:text-red-500" onclick="deleteService(${s.id}, true)"><i class="fas fa-trash"></i></button></td>
                                </tr>
                            `).join('')}
                        </tbody>
                    </table>
                ` : '<p class="text-center p-4">سرویسی یافت نشد.</p>';
                renderPagination('services-pagination', data.current_page, data.total_pages, (newPage) => loadServicesList(newPage, search, server, status));
             } else {
                 container.innerHTML = '<p class="text-red-500 text-center p-4">خطا در بارگذاری سرویس‌ها.</p>';
        
        function renderUserDetailsInto(containerId, data) {
            const container = document.getElementById(containerId);
            if (!data.success) {
                container.innerHTML = `<p class="text-red-400 text-center p-4">${data.message}</p>`;
                return;
            const user = data.user;
            const services = (user.services || []);
            const uniqueServers = [...new Set(services.map(s => s.location).filter(Boolean))];
            let servicesHtml = '<p class="mt-4 px-3">این کاربر سرویسی ندارد.</p>';

            if (services.length > 0) {
                servicesHtml = services.map(s => `
                    <div class="bg-white/5 border border-white/10 rounded-xl p-5 shadow-sm transition hover:shadow-md" data-status="${s.details.is_active ? 'active' : 'inactive'}" data-server="${s.location || ''}" data-text="${(s.display||'')} ${(s.name||'')} ${(s.uuid||'')}">
                        <div class="flex justify-between items-start mb-4">
                            <div class="flex-grow">
                                <p class="font-bold text-lg">${s.display}</p>
                                <p class="text-xs text-slate-400 font-mono">${s.name} | ${s.uuid}</p>
                            </div>
                            <span class="px-3 py-1 text-xs font-semibold rounded-full ${s.details.is_active ? 'bg-green-500/20 text-green-300' : 'bg-red-500/20 text-red-300'}">
                                ${s.details.is_active ? 'فعال' : 'غیرفعال'}
                            </span>
                        </div>
                        <div>
                            <div class="flex justify-between items-center mb-1 text-sm">
                                <span class="font-medium">مصرف ترافیک</span>
                                ${s.details.used_traffic !== null ? `<span class="font-bold">${s.details.used_traffic} / ${s.size} GB</span>` : `<span class="text-slate-400">نامشخص / ${s.size} GB</span>`}
                            </div>
                            <div class="w-full bg-slate-700 rounded-full h-2.5">
                                <div class="bg-gradient-to-r from-sky-500 to-indigo-500 h-2.5 rounded-full" style="width: ${s.details.progress_percentage}%"></div>
                            </div>
                        </div>
                        <div class="grid grid-cols-2 gap-4 text-sm mt-5 pt-4 border-t border-white/10">
                            <div class="flex items-center"><i class="fas fa-server ml-2 text-slate-400"></i><strong>سرور:</strong><span class="mr-2">${s.location}</span></div>
                            <div class="flex items-center"><i class="fas fa-calendar-plus ml-2 text-slate-400"></i><strong>تاریخ ساخت:</strong><span class="mr-2">${s.details.created_at_jalali}</span></div>
                            <div class="flex items-center"><i class="fas fa-hourglass-half ml-2 text-slate-400"></i><strong>روز مانده:</strong><span class="mr-2">${s.details.days_remaining} روز</span></div>
                            <div class="flex items-center"><i class="fas fa-calendar-times ml-2 text-slate-400"></i><strong>تاریخ انقضا:</strong><span class="mr-2">${s.details.expire_date_jalali}</span></div>
                        </div>
                        <div class="text-left mt-4">
                           <button class="text-red-400 hover:text-red-500 text-sm" onclick="deleteService(${s.id}, false)"><i class="fas fa-trash-alt ml-1"></i>حذف سرویس</button>
                        </div>
                    </div>
                `).join('');

            container.innerHTML = `
                <div class="space-y-6">
                    <div class="bg-white/10 p-6 rounded-xl border border-white/10 shadow-sm">
                       <div class="grid grid-cols-1 md:grid-cols-2 gap-x-6 gap-y-4">
                            <div class="flex items-center"><i class="fas fa-id-card text-lg ml-3 text-slate-400"></i><p><strong>ID کاربر:</strong> <code class="bg-slate-700 p-1 rounded-md text-sm">${user.id}</code></p></div>
                            <div class="flex items-center"><i class="fab fa-telegram-plane text-lg ml-3 text-slate-400"></i><p><strong>نام کاربری:</strong> ${user.username ? '@' + user.username : 'ندارد'}</p></div>
                       </div>
                        <div class="border-t border-white/10 pt-4 mt-4">
                            <p class="font-semibold text-lg flex items-center"><i class="fas fa-wallet text-lg ml-3 text-slate-400"></i><strong>موجودی:</strong><span class="mr-2 text-xl font-bold text-green-400">${new Intl.NumberFormat('fa-IR').format(user.coin)}</span><span class="text-sm mr-1">تومان</span></p>
                            <div class="flex items-center space-x-2 space-x-reverse mt-3">
                                <input type="number" id="balance-amount-${user.id}" class="p-2 w-40" placeholder="مبلغ به تومان">
                                <button class="bg-green-600 text-white px-3 py-2 rounded-lg text-sm transition hover:bg-green-700" onclick="updateBalance('${user.id}', 'add')">افزایش</button>
                                <button class="bg-red-600 text-white px-3 py-2 rounded-lg text-sm transition hover:bg-red-700" onclick="updateBalance('${user.id}', 'subtract')">کاهش</button>
                            </div>
                        </div>
                    </div>
                    <div>
                        <h4 class="font-bold text-2xl mb-4">سرویس‌های کاربر (${user.services.length})</h4>
                        <div id="${containerId}-services-list" class="space-y-5">${servicesHtml}</div>
                    </div>
                </div>
            `;
        function displayUserDetails(data) { renderUserDetailsInto('user-details-container', data); }
        function displayUserDetailsPanel(data) { openUserDrawer(); renderUserDetailsInto('user-drawer-body', data); }
        
        // =================================================================
        // ===== ACTIONS & EVENT HANDLERS =====
        // =================================================================

        // --- Search/Filter Timers ---
        let userSearchTimer, serviceSearchTimer, messagesSearchTimer, userDetailsSearchTimer;
        document.getElementById('users-list-search').addEventListener('keyup', function() {
            clearTimeout(userSearchTimer);
            userSearchTimer = setTimeout(() => { loadUsersList(1, this.value, document.getElementById('users-list-status-filter')?.value || '', document.getElementById('users-list-server-filter')?.value || ''); }, 500);
        });
        
        function setupServiceFilters() {
            const searchInput = document.getElementById('services-list-search');
            const serverFilter = document.getElementById('services-list-server-filter');
            const statusFilter = document.getElementById('services-list-status-filter');
            const applyFilters = () => {
                clearTimeout(serviceSearchTimer);
                serviceSearchTimer = setTimeout(() => {
                    loadServicesList(1, searchInput.value, serverFilter.value, statusFilter.value);
                }, 500);
            };
            ['keyup', 'change'].forEach(evt => {
                searchInput.addEventListener(evt, applyFilters);
                serverFilter.addEventListener(evt, applyFilters);
                statusFilter.addEventListener(evt, applyFilters);
            });
            apiRequest('get_servers').then(data => {
                if(data.success) {
                    serverFilter.innerHTML = '<option value="">همه سرورها</option>' + data.servers.map(s => `<option value="${s.name}">${s.display_name}</option>`).join('');
            });
        setupServiceFilters();

        function setupUsersListFilters() {
            const statusEl = document.getElementById('users-list-status-filter');
            const serverEl = document.getElementById('users-list-server-filter');
            if (!statusEl || !serverEl) return;
            const apply = () => {
                const q = document.getElementById('users-list-search')?.value || '';
                loadUsersList(1, q, statusEl.value, serverEl.value);
            };
            statusEl.addEventListener('change', apply);
            serverEl.addEventListener('change', apply);
            apiRequest('get_servers').then(d => {
                if (d.success && d.servers) {
                    serverEl.innerHTML = '<option value="">همه سرورها</option>' + d.servers.map(s => `<option value="${s.name}">${s.display_name}</option>`).join('');
            });
        setupUsersListFilters();

        // Generic filters for user services inside details containers (drawer or search page)
        const _userFilterTimers = {};
        function applyUserServiceFilters(baseId) {
            const statusEl = document.getElementById(`${baseId}-status-filter`);
            const serverEl = document.getElementById(`${baseId}-server-filter`);
            const searchEl = document.getElementById(`${baseId}-search`);
            const listEl = document.getElementById(`${baseId}-services-list`);
            if (!listEl) return;
            const st = statusEl?.value || '';
            const sv = serverEl?.value || '';
            const q = (searchEl?.value || '').toLowerCase();
            const cards = listEl.children || [];
            for (let i = 0; i < cards.length; i++) {
                const el = cards[i];
                const okStatus = !st || (el.getAttribute('data-status') === st);
                const okServer = !sv || (el.getAttribute('data-server') === sv);
                const okSearch = !q || (el.getAttribute('data-text') || '').toLowerCase().includes(q);
                el.classList.toggle('hidden', !(okStatus && okServer && okSearch));
        document.addEventListener('change', function(e) {
            const t = e.target; if (!t || !t.id) return;
            if (t.id.endsWith('-status-filter') || t.id.endsWith('-server-filter')) {
                const baseId = t.id.replace(/-(status|server)-filter$/, '');
                applyUserServiceFilters(baseId);
        });
        document.addEventListener('keyup', function(e) {
            const t = e.target; if (!t || !t.id) return;
            if (t.id.endsWith('-search')) {
                const baseId = t.id.substring(0, t.id.length - '-search'.length);
                clearTimeout(_userFilterTimers[baseId]);
                _userFilterTimers[baseId] = setTimeout(() => applyUserServiceFilters(baseId), 250);
        });

        // Payments UI: show/hide detail sections based on toggles
        function updatePaymentSectionsVisibility() {
            const cardOn = document.getElementById('card-payment-status-toggle')?.checked;
            const cryptoOn = document.getElementById('crypto-payment-status-toggle')?.checked;
            const kycOn = document.getElementById('kyc-status-toggle')?.checked;
            const cardDetails = document.getElementById('card-payment-details');
            const cryptoDetails = document.getElementById('crypto-payment-details');
            const kycDetails = document.getElementById('kyc-details');
            if (cardDetails) cardDetails.classList.toggle('hidden', !cardOn);
            if (cryptoDetails) cryptoDetails.classList.toggle('hidden', !cryptoOn);
            if (kycDetails) kycDetails.classList.toggle('hidden', !kycOn);
        ['card-payment-status-toggle','crypto-payment-status-toggle','kyc-status-toggle'].forEach(id => {
            const el = document.getElementById(id);
            el?.addEventListener('change', updatePaymentSectionsVisibility);
        });
        // Initialize visibility on load
        updatePaymentSectionsVisibility();

        document.getElementById('messages-search').addEventListener('keyup', function() {
            clearTimeout(messagesSearchTimer);
            messagesSearchTimer = setTimeout(() => { loadMessages(this.value); }, 500);
        });

        const userSearchInput = document.getElementById('user-search-input');
        userSearchInput.addEventListener('keyup', function() {
            clearTimeout(userDetailsSearchTimer);
            const container = document.getElementById('user-details-container');
            userDetailsSearchTimer = setTimeout(async () => {
                const query = this.value.trim();
                if (query.length < 3) {
                    container.innerHTML = '<p class="text-center py-8">برای جستجو حداقل ۳ کاراکتر وارد کنید.</p>';
                    return;
                showLoader(container);
                const data = await apiRequest('get_user_details', { identifier: query });
                displayUserDetails(data);
            }, 500);
        });
        
        // --- User Actions ---
        window.updateBalance = async function(userId, type) {
            const amount = document.getElementById(`balance-amount-${userId}`).value;
            if (!amount || amount <= 0) { openModal('<p>لطفا مبلغ معتبری وارد کنید.</p>'); return; }
            const result = await apiRequest('update_balance', { user_id: userId, amount: amount, type: type });
            openModal(`<p>${result.success ? 'موجودی با موفقیت بروزرسانی شد.' : 'خطا: ' + result.message}</p>`);
            if (result.success) userSearchInput.dispatchEvent(new Event('keyup'));

        window.deleteService = async function(serviceId, refreshList) {
            if (!confirm('آیا از حذف این سرویس از ربات و سرور مطمئن هستید؟ این عمل غیرقابل بازگشت است.')) return;
            const result = await apiRequest('delete_service', { service_id: serviceId });
            openModal(`<p>${result.success ? 'سرویس با موفقیت حذف شد.' : 'خطا: ' + result.message}</p>`);
            if(result.success) {
                 if (refreshList) {
                    loadServicesList(1, document.getElementById('services-list-search').value, document.getElementById('services-list-server-filter').value, document.getElementById('services-list-status-filter').value);
                } else {
                    userSearchInput.dispatchEvent(new Event('keyup'));

        // =================================================================
        // ===== SERVER MANAGEMENT FUNCTIONS (Corrected from old version) =====
        // =================================================================

        async function loadServers() {
            const serverList = document.getElementById('server-list');
            showLoader(serverList);
            const data = await apiRequest('get_servers');
            if (data.success) {
                if (data.servers.length === 0) {
                    serverList.innerHTML = '<p class="text-center p-4">هیچ سروری یافت نشد.</p>';
                    return;
                const serverMap = {};
                data.servers.forEach(s => serverMap[s.id] = s);
                window.serverMap = serverMap;
                const table = `
                    <table class="w-full text-right table-cool">
                        <thead>
                            <tr>
                                <th class="p-3">نام نمایشی</th>
                                <th class="p-3">نوع</th>
                                <th class="p-3">وضعیت</th>
                                <th class="p-3">عملیات</th>
                            </tr>
                        </thead>
                        <tbody>
                            ${data.servers.map(server => `
                                <tr>
                                    <td class="p-4">${server.display_name}</td>
                                    <td class="p-4">${server.type}</td>
                                    <td class="p-4">${server.status === 'on' ? '<span class="px-2 py-1 bg-green-500/20 text-green-300 rounded-full text-xs font-medium">روشن</span>' : '<span class="px-2 py-1 bg-red-500/20 text-red-300 rounded-full text-xs font-medium">خاموش</span>'}</td>
                                    <td class="p-4 space-x-2 space-x-reverse">
                                        <button class="btn-icon" onclick='openEditServerModal(${server.id})' title="ویرایش"><i class="fas fa-gear"></i></button>
                                        <button class="btn-icon" onclick='toggleServerStatus(${server.id})' title="${server.status === 'on' ? 'خاموش کردن' : 'روشن کردن'}"><i class="fas fa-power-off"></i></button>
                                        <button class="btn-icon" onclick='toggleServerHidden(${server.id})' title="${server.is_hidden == 1 ? 'نمایش در لیست خرید' : 'مخفی کردن از لیست خرید'}"><i class="fas ${server.is_hidden == 1 ? 'fa-eye-slash' : 'fa-eye'}"></i></button>
                                        <button class="btn-icon ${server.is_test_server == 1 ? 'text-green-400' : ''}" onclick='setTestServer(${server.id})' title="انتخاب به عنوان سرور تست"><i class="fas fa-vial-circle-check"></i></button>
                                        <button class="btn-icon text-red-400 hover:text-red-500" onclick='deleteServer(${server.id})' title="حذف"><i class="fas fa-trash"></i></button>
                                    </td>
                                </tr>
                            `).join('')}
                        </tbody>
                    </table>
                `;
                serverList.innerHTML = table;
            } else {
                serverList.innerHTML = '<p class="text-red-500 text-center">خطا در بارگذاری سرورها.</p>';
        window.loadServers = loadServers;

        window.toggleServerStatus = async function(serverId) {
            const result = await apiRequest('toggle_server_status', { server_id: serverId });
            if (result.success) { loadServers(); } else { alert('خطا در تغییر وضعیت سرور.'); }
        window.toggleServerHidden = async function(serverId) {
            const result = await apiRequest('toggle_server_hidden', { server_id: serverId });
            if (result.success) { loadServers(); } else { alert('خطا در تغییر وضعیت نمایش سرور.'); }
        window.setTestServer = async function(serverId) {
            const result = await apiRequest('set_test_server', { server_id: serverId });
            if (result.success) { loadServers(); } else { alert('خطا در انتخاب سرور تست.'); }
        window.deleteServer = async function(serverId) {
            if (!confirm('آیا از حذف این سرور مطمئن هستید؟')) return;
            const result = await apiRequest('delete_server', { server_id: serverId });
            if (result.success) { loadServers(); } else { alert('خطا در حذف سرور.'); }

        window.openEditServerModal = function(serverId) {
            const server = window.serverMap[serverId];
            if (!server) return;
            const modalHtml = `
                <h3 class="text-xl font-bold mb-6 border-b pb-3 border-slate-600">ویرایش سرور</h3>
                <form id="edit-server-form" class="space-y-4 text-sm">
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div><label class="font-medium">نام سیستمی:</label><input type="text" name="name" class="w-full p-2 mt-1" value="${server.name}" required></div>
                        <div><label class="font-medium">نام نمایشی:</label><input type="text" name="display_name" class="w-full p-2 mt-1" value="${server.display_name}" required></div>
                    </div>
                    <div>
                        <label class="font-medium">نوع سرور:</label>
                        <select name="type" class="w-full p-2 bg-slate-700 mt-1" required>
                            <option value="marzban" ${server.type === 'marzban' ? 'selected' : ''}>Marzban</option>
                            <option value="sanaei" ${server.type === 'sanaei' ? 'selected' : ''}>Sanaei</option>
                            <option value="marzneshin" ${server.type === 'marzneshin' ? 'selected' : ''}>Marzneshin</option>
                        </select>
                    </div>
                    <div><label class="font-medium">لینک پنل:</label><input type="text" name="link" class="w-full p-2 mt-1" value="${server.link}" required></div>
                    <div>
                        <label class="font-medium">دامنه جایگزین اشتراک (اختیاری):</label>
                        <input type="text" name="custom_domain" class="w-full p-2 mt-1" value="${server.custom_domain || ''}" placeholder="sub.domain.com یا domain1.com|domain2.com">
                        <label class="flex items-center gap-2 mt-2 text-xs text-slate-300"><input type="checkbox" name="custom_domain_pool_enabled" value="1" ${String(server.custom_domain_pool_enabled) === '1' ? 'checked' : ''}><span>توزیع خودکار دامنه بین کاربران</span></label>
                        <p class="text-xs text-slate-400 mt-1">دامنه‌ها را با کاراکتر <code>|</code> جدا کنید تا برای هر سرویس جدید یکی از آن‌ها اختصاص یابد.</p>
                    </div>
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div><label class="font-medium">یوزرنیم (اختیاری):</label><input type="text" name="username" class="w-full p-2 mt-1" value="${server.username || ''}"></div>
                        <div><label class="font-medium">پسورد (اختیاری):</label><input type="text" name="password" class="w-full p-2 mt-1" value="${server.password || ''}"></div>
                    </div>
                    <div class="border-t pt-4 border-slate-600">
                        <p class="font-semibold text-slate-300 mb-2">تنظیمات اختصاصی هر پنل:</p>
                        <div data-server-type="marzban"><label class="font-medium">پروتکل‌ها (برای مرزبان، با | جدا شود):</label><input type="text" name="protocols" class="w-full p-2 mt-1" value="${server.protocols || ''}"></div>
                        <div data-server-type="marzneshin"><label class="font-medium">شناسه سرویس‌ها (برای مرزنشین، با , یا | جدا شود):</label><input type="text" name="marzneshin_service_ids" class="w-full p-2 mt-1" value="${server.marzneshin_service_ids || ''}"></div>
                        <div data-server-type="sanaei"><label class="font-medium">ID اینباند کپی (برای ثنایی):</label><input type="text" name="inbound_copy" class="w-full p-2 mt-1" value="${server.inbound_copy || ''}"></div>
                        <div data-server-type="sanaei"><label class="font-medium">لینک نمونه (برای ثنایی):</label><input type="text" name="example_link" class="w-full p-2 mt-1" value="${server.example_link || ''}"></div>
                    </div>
                    <div class="flex justify-end space-x-3 space-x-reverse pt-6">
                        <button type="button" onclick="closeModal()" class="px-5 py-2 rounded-lg transition">انصراف</button>
                        <button type="submit" class="bg-indigo-600 hover:bg-indigo-700 text-white px-5 py-2 rounded-lg transition">ذخیره</button>
                    </div>
                </form>
            `;
            openModal(modalHtml);
            if (typeof applyServerTypeVisibility === 'function') {
                applyServerTypeVisibility(document.getElementById('edit-server-form'));
            }
            document.getElementById('edit-server-form').addEventListener('submit', async function(e) {
                e.preventDefault();
                const formData = new FormData(this);
                formData.append('server_id', serverId);
                const params = Object.fromEntries(formData.entries());
                const result = await apiRequest('update_server', params);
                if (result.success) {
                    alert('سرور با موفقیت بروزرسانی شد.');
                    closeModal();
                    loadServers();
                } else {
                    alert(`خطا: ${result.message}`);
            });

        document.getElementById('add-server-btn').addEventListener('click', () => {
            const modalHtml = `
                <h3 class="text-xl font-bold mb-6 border-b pb-3 border-slate-600">افزودن سرور جدید</h3>
                <form id="add-server-form" class="space-y-4 text-sm">
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div><label class="font-medium">نام سیستمی (مثلا S1Germany):</label><input type="text" name="name" class="w-full p-2 mt-1" required></div>
                        <div><label class="font-medium">نام نمایشی (برای ربات):</label><input type="text" name="display_name" class="w-full p-2 mt-1" required></div>
                    </div>
                    <div>
                        <label class="font-medium">نوع سرور:</label>
                        <select name="type" class="w-full p-2 bg-slate-700 mt-1" required>
                            <option value="marzban">Marzban</option>
                            <option value="sanaei">Sanaei</option>
                            <option value="marzneshin">Marzneshin</option>
                        </select>
                    </div>
                    <div><label class="font-medium">لینک پنل:</label><input type="text" name="link" class="w-full p-2 mt-1" required></div>
                    <div>
                        <label class="font-medium">دامنه جایگزین اشتراک (اختیاری):</label>
                        <input type="text" name="custom_domain" class="w-full p-2 mt-1" placeholder="sub.domain.com یا domain1.com|domain2.com">
                        <label class="flex items-center gap-2 mt-2 text-xs text-slate-300"><input type="checkbox" name="custom_domain_pool_enabled" value="1"><span>توزیع خودکار دامنه بین کاربران (چند دامنه را با | جدا کنید)</span></label>
                        <p class="text-xs text-slate-400 mt-1">با فعال‌سازی، دامنه‌ها به‌صورت ترتیبی بین سرویس‌های جدید تقسیم می‌شوند.</p>
                    </div>
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div><label class="font-medium">یوزرنیم (اختیاری):</label><input type="text" name="username" class="w-full p-2 mt-1"></div>
                        <div><label class="font-medium">پسورد (اختیاری):</label><input type="text" name="password" class="w-full p-2 mt-1"></div>
                    </div>
                    <div class="border-t pt-4 border-slate-600">
                        <p class="font-semibold text-slate-300 mb-2">تنظیمات اختصاصی هر پنل:</p>
                        <div data-server-type="marzban"><label class="font-medium">پروتکل‌ها (برای مرزبان، با | جدا شود):</label><input type="text" name="protocols" placeholder="vless|vmess|trojan" class="w-full p-2 mt-1"></div>
                        <div data-server-type="marzneshin"><label class="font-medium">شناسه سرویس‌ها (برای مرزنشین، با , یا | جدا شود):</label><input type="text" name="marzneshin_service_ids" class="w-full p-2 mt-1" placeholder="مثال: 1,2,5"></div>
                        <div data-server-type="sanaei"><label class="font-medium">ID اینباند کپی (برای ثنایی):</label><input type="text" name="inbound_copy" class="w-full p-2 mt-1"></div>
                        <div data-server-type="sanaei"><label class="font-medium">لینک نمونه (برای ثنایی):</label><input type="text" name="example_link" class="w-full p-2 mt-1"></div>
                    </div>
                    <div class="flex justify-end space-x-3 space-x-reverse pt-6">
                        <button type="button" onclick="closeModal()" class="px-5 py-2 rounded-lg transition">انصراف</button>
                        <button type="submit" class="bg-indigo-600 hover:bg-indigo-700 text-white px-5 py-2 rounded-lg transition">افزودن</button>
                    </div>
                </form>
            `;
            openModal(modalHtml);
            if (typeof applyServerTypeVisibility === 'function') {
                applyServerTypeVisibility(document.getElementById('add-server-form'));
            }

            document.getElementById('add-server-form').addEventListener('submit', async function(e) {
                e.preventDefault();
                const formData = new FormData(this);
                const params = Object.fromEntries(formData.entries());
                const result = await apiRequest('add_server', params);
                if (result.success) {
                    alert('سرور با موفقیت اضافه شد.');
                    closeModal();
                    loadServers();
                } else {
                    alert(`خطا: ${result.message}`);
            });
        });

        // --- Plan Actions ---
        async function loadPlans() {
            const planList = document.getElementById('plan-list');
            showLoader(planList);
            const data = await apiRequest('get_plans');
            if (data.success) {
                if (data.plans.length === 0) {
                    planList.innerHTML = '<p class="text-center p-4">هیچ تعرفه‌ای یافت نشد.</p>';
                    return;
                const table = `
                    <table class="w-full text-right table-cool">
                        <thead>
                            <tr>
                                <th class="p-3">عنوان</th>
                                <th class="p-3">حجم (GB)</th>
                                <th class="p-3">مدت (روز)</th>
                                <th class="p-3">عملیات</th>
                            </tr>
                        </thead>
                        <tbody>
                            ${data.plans.map(plan => `
                                <tr>
                                    <td class="p-4">${plan.title}</td>
                                    <td class="p-4">${plan.volume}</td>
                                    <td class="p-4">${plan.duration}</td>
                                    <td class="p-4">
                                        <button class="btn-icon text-red-400 hover:text-red-500" onclick='deletePlan(${plan.id})' title="حذف"><i class="fas fa-trash"></i></button>
                                    </td>
                                </tr>
                            `).join('')}
                        </tbody>
                    </table>
                `;
                planList.innerHTML = table;
            } else {
                planList.innerHTML = '<p class="text-red-500 text-center">خطا در بارگذاری تعرفه‌ها.</p>';
        window.loadPlans = loadPlans;

        window.deletePlan = async function(planId) {
             if (!confirm('آیا از حذف این تعرفه مطمئن هستید؟ قیمت‌های مرتبط با آن نیز حذف خواهند شد.')) return;
            const result = await apiRequest('delete_plan', { plan_id: planId });
            if (result.success) { loadPlans(); } else { alert('خطا در حذف تعرفه.'); }
        document.getElementById('add-plan-btn').addEventListener('click', () => {
            const modalHtml = `
                <h3 class="text-xl font-bold mb-6 border-b pb-3 border-slate-600">افزودن تعرفه جدید</h3>
                <form id="add-plan-form" class="space-y-4">
                    <div><label class="font-medium">عنوان تعرفه:</label><input type="text" name="title" class="w-full p-2 mt-1" required></div>
                    <div><label class="font-medium">حجم (GB):</label><input type="number" name="volume" class="w-full p-2 mt-1" required></div>
                    <div><label class="font-medium">مدت (روز):</label><input type="number" name="duration" class="w-full p-2 mt-1" required></div>
                    <div class="flex justify-end space-x-3 space-x-reverse pt-6">
                        <button type="button" onclick="closeModal()" class="px-5 py-2 rounded-lg transition">انصراف</button>
                        <button type="submit" class="bg-indigo-600 hover:bg-indigo-700 text-white px-5 py-2 rounded-lg transition">افزودن</button>
                    </div>
                </form>
            `;
            openModal(modalHtml);
            document.getElementById('add-plan-form').addEventListener('submit', async function(e) {
                e.preventDefault();
                const formData = new FormData(this);
                const params = Object.fromEntries(formData.entries());
                const result = await apiRequest('add_plan', params);
                if (result.success) {
                    alert('تعرفه با موفقیت اضافه شد.');
                    closeModal();
                    loadPlans();
                } else {
                    alert(`خطا: ${result.message}`);
            });
        });
        
        // --- Pricing Actions ---
        async function loadPricing() {
            const serverSelect = document.getElementById('pricing-server-select');
            const priceList = document.getElementById('price-list');
            serverSelect.innerHTML = '<option>در حال بارگذاری سرورها...</option>';
            const data = await apiRequest('get_servers');
            if (data.success && data.servers.length > 0) {
                serverSelect.innerHTML = '<option value="">-- انتخاب کنید --</option>';
                data.servers.forEach(server => {
                    if (server.is_test_server != 1) {
                        serverSelect.innerHTML += `<option value="${server.id}">${server.display_name}</option>`;
                });
                priceList.innerHTML = '<p class="text-center p-4">لطفا یک سرور را انتخاب کنید.</p>';
            } else {
                 serverSelect.innerHTML = '<option>هیچ سروری یافت نشد</option>';
                 priceList.innerHTML = '';

        document.getElementById('pricing-server-select').addEventListener('change', async function() {
            const serverId = this.value;
            const priceList = document.getElementById('price-list');
            if (!serverId) { priceList.innerHTML = ''; return; }
            showLoader(priceList);
            const data = await apiRequest('get_prices', { server_id: serverId });
            if (data.success) {
                if (data.prices.length === 0) {
                    priceList.innerHTML = '<p class="text-center p-4">هیچ تعرفه‌ای برای قیمت‌گذاری یافت نشد.</p>';
                    return;
                }
                priceList.innerHTML = data.prices.map(item => `
                    <div class="flex items-center justify-between border-b border-slate-700 py-4">
                        <span class="font-medium">${item.title} (${item.volume}GB, ${item.duration} روز)</span>
                        <div class="flex items-center">
                            <input type="number" class="w-32 p-2 text-left" id="price-input-${item.plan_id}" value="${item.price || ''}" placeholder="قیمت به تومان">
                            <input type="number" class="w-28 p-2 text-left ml-2" id="stars-input-${item.plan_id}" value="${item.stars_price || ''}" placeholder="استارز">
                            <button class="bg-indigo-600 text-white px-4 py-2 rounded-lg mr-2 transition hover:bg-indigo-700" onclick="setPrice(${serverId}, ${item.plan_id})">ذخیره</button>
                        </div>
                    </div>
                `).join('');
            } else {
                priceList.innerHTML = '<p class="text-red-500 text-center">خطا در دریافت قیمت‌ها.</p>';
            }
        });

        window.setPrice = async function(serverId, planId) {
            const priceInput = document.getElementById(`price-input-${planId}`);
            const starsInput = document.getElementById(`stars-input-${planId}`);
            const price = priceInput.value === '' ? 0 : parseInt(priceInput.value, 10);
            const stars = starsInput.value === '' ? 0 : parseInt(starsInput.value, 10);
            if (price < 0 || stars < 0) { alert('لطفا مقدار معتبری وارد کنید.'); return; }
            const result = await apiRequest('set_price', { server_id: serverId, plan_id: planId, price: price, stars_price: stars });
            if (result.success) {
                [priceInput, starsInput].forEach(el => { el.classList.add('!bg-green-500/20', '!border-green-500'); setTimeout(() => el.classList.remove('!bg-green-500/20', '!border-green-500'), 2000); });
            } else {
                alert(`خطا: ${result.message}`);
                [priceInput, starsInput].forEach(el => { el.classList.add('!bg-red-500/20', '!border-red-500'); setTimeout(() => el.classList.remove('!bg-red-500/20', '!border-red-500'), 2000); });
            }
        };

        // --- Discount Actions ---
        async function loadDiscounts() {
            const container = document.getElementById('discounts-list');
            showLoader(container);
            const data = await apiRequest('get_discounts');
            if (!data.success) { container.innerHTML = '<div class="p-4 text-red-400">خطا در دریافت لیست کدها</div>'; return; }
            container.innerHTML = `
              <table class="min-w-full text-sm table-cool">
                <thead>
                  <tr>
                    <th class="px-3 py-2 text-right">کد</th>
                    <th class="px-3 py-2 text-right">نوع</th>
                    <th class="px-3 py-2 text-right">مقدار</th>
                    <th class="px-3 py-2 text-right">مصرف</th>
                    <th class="px-3 py-2 text-right">سقف کاربر</th>
                    <th class="px-3 py-2 text-right">حداقل مبلغ</th>
                    <th class="px-3 py-2 text-right">انقضا</th>
                    <th class="px-3 py-2 text-right">وضعیت</th>
                    <th class="px-3 py-2 text-right">اقدامات</th>
                  </tr>
                </thead>
                <tbody>${data.discounts.map(item => `
                    <tr>
                      <td class="px-3 py-2 font-mono">${item.code}</td>
                      <td class="px-3 py-2">${item.type === 'percent' ? 'درصدی' : 'مبلغی'}</td>
                      <td class="px-3 py-2">${item.type === 'percent' ? (item.value + '%') : (Number(item.value).toLocaleString('fa-IR') + ' تومان')}</td>
                      <td class="px-3 py-2">${(item.used_count ?? 0)} / ${item.max_uses ?? '∞'}</td>
                      <td class="px-3 py-2">${item.per_user_limit ?? '-'}</td>
                      <td class="px-3 py-2">${item.min_order_amount ? Number(item.min_order_amount).toLocaleString('fa-IR') : '-'}</td>
                      <td class="px-3 py-2">${item.expires_at ?? '-'}</td>
                      <td class="px-3 py-2">${item.is_active == 1 ? '<span class="text-green-400">فعال</span>' : '<span class="text-slate-400">غیرفعال</span>'}</td>
                      <td class="px-3 py-2 flex gap-2">
                        <button class="px-3 py-1 text-xs" onclick="toggleDiscount(${item.id})">تغییر وضعیت</button>
                        <button class="px-3 py-1 text-xs" onclick='showEditDiscountModal(${JSON.stringify(item)})'>ویرایش</button>
                        <button class="px-3 py-1 text-xs" onclick="deleteDiscount(${item.id})">حذف</button>
                      </td>
                    </tr>
                `).join('')}</tbody>
              </table>`;
            document.getElementById('add-discount-btn').onclick = showAddDiscountModal;
        
        function discountFormHTML(values = {}) {
            const v = Object.assign({id:null, code:'', type:'percent', value:'', max_uses:'', per_user_limit:'', min_order_amount:'', expires_at:'', is_active:1, note:''}, values);
            return `
              <h3 class="text-xl font-bold mb-6 border-b pb-3 border-slate-600">${v.id ? 'ویرایش کد تخفیف' : 'افزودن کد تخفیف'}</h3>
              <form id="discount-form" class="space-y-4">
                <div><label class="font-medium text-sm">کد:</label><input type="text" name="code" class="w-full p-2 mt-1" required value="${v.code}"></div>
                <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                  <div><label class="font-medium text-sm">نوع:</label>
                    <select name="type" class="w-full p-2 mt-1">
                      <option value="percent" ${v.type==='percent'?'selected':''}>درصدی</option>
                      <option value="fixed" ${v.type==='fixed'?'selected':''}>مبلغی</option>
                    </select>
                  </div>
                  <div><label class="font-medium text-sm">مقدار:</label><input type="number" step="0.01" name="value" class="w-full p-2 mt-1" required value="${v.value}"></div>
                  <div><label class="font-medium text-sm">حداقل مبلغ سفارش:</label><input type="number" name="min_order_amount" class="w-full p-2 mt-1" value="${v.min_order_amount ?? ''}"></div>
                </div>
                <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                  <div><label class="font-medium text-sm">سقف دفعات (کل):</label><input type="number" name="max_uses" class="w-full p-2 mt-1" value="${v.max_uses ?? ''}"></div>
                  <div><label class="font-medium text-sm">سقف برای هر کاربر:</label><input type="number" name="per_user_limit" class="w-full p-2 mt-1" value="${v.per_user_limit ?? ''}"></div>
                  <div><label class="font-medium text-sm">انقضا:</label><input type="datetime-local" name="expires_at" class="w-full p-2 mt-1" value="${v.expires_at ? v.expires_at.replace(' ', 'T') : ''}"></div>
                </div>
                <div class="flex items-center gap-2"><input type="checkbox" name="is_active" ${Number(v.is_active)===1?'checked':''}><span>فعال</span></div>
                <div><label class="font-medium text-sm">یادداشت:</label><input type="text" name="note" class="w-full p-2 mt-1" value="${v.note ?? ''}"></div>
                <div class="flex justify-end gap-2 pt-4">
                  <button type="button" onclick="closeModal()">انصراف</button>
                  <button type="submit">ثبت</button>
                </div>
              </form>`;
        function showAddDiscountModal() {
            openModal(discountFormHTML());
            document.getElementById('discount-form').addEventListener('submit', async (e) => {
                e.preventDefault();
                const params = Object.fromEntries(new FormData(e.target).entries());
                const res = await apiRequest('create_discount', params);
                if (res.success) { closeModal(); loadDiscounts(); } else { alert(res.message || 'خطا'); }
            });
        function showEditDiscountModal(item) {
            openModal(discountFormHTML(item));
            document.getElementById('discount-form').addEventListener('submit', async (e) => {
                e.preventDefault();
                const params = Object.fromEntries(new FormData(e.target).entries());
                params.id = item.id;
                const res = await apiRequest('update_discount', params);
                if (res.success) { closeModal(); loadDiscounts(); } else { alert(res.message || 'خطا'); }
            });
        async function deleteDiscount(id) {
            if (!confirm('حذف این کد؟')) return;
            const res = await apiRequest('delete_discount', { id });
            if (res.success) loadDiscounts(); else alert('خطا در حذف');
        async function toggleDiscount(id) {
            const res = await apiRequest('toggle_discount_status', { id });
            if (res.success) loadDiscounts(); else alert('خطا در تغییر وضعیت');

        // Expose discount actions for inline handlers
        window.loadDiscounts = loadDiscounts;
        window.showAddDiscountModal = showAddDiscountModal;
        window.showEditDiscountModal = showEditDiscountModal;
        window.deleteDiscount = deleteDiscount;
        window.toggleDiscount = toggleDiscount;

        // --- Message Actions ---
        async function loadMessages(search = '') {
            const container = document.getElementById('messages-list');
            showLoader(container);
            const data = await apiRequest('get_messages', { search });
            if (data.success) {
                container.innerHTML = data.messages.map(msg => `
                    <div class="border border-slate-700 rounded-lg mb-2 overflow-hidden shadow-sm">
                        <div class="bg-white/5 p-4 cursor-pointer flex justify-between items-center transition hover:bg-white/10" onclick="this.nextElementSibling.classList.toggle('hidden'); this.querySelector('i').classList.toggle('rotate-180')">
                            <div>
                                <h4 class="font-semibold">${msg.message_key}</h4>
                                <p class="text-sm text-slate-400">${msg.description || 'بدون توضیحات'}</p>
                            </div>
                            <i class="fas fa-chevron-down text-slate-400 transition-transform"></i>
                        </div>
                        <div class="p-4 hidden bg-white/5 border-t border-slate-700">
                            <textarea id="msg-val-${msg.message_key}" class="w-full p-2 mb-2" rows="4">${msg.message_value}</textarea>
                            <div class="flex justify-end">
                                <button class="bg-indigo-600 hover:bg-indigo-700 text-white px-5 py-2 rounded-lg font-semibold transition" onclick="updateMessage('${msg.message_key}')">ذخیره</button>
                            </div>
                        </div>
                    </div>
                `).join('') || '<p class="text-center p-4">هیچ پیامی یافت نشد.</p>';
            } else {
                container.innerHTML = '<p class="text-red-500 text-center p-4">خطا در بارگذاری پیام‌ها.</p>';
        window.loadMessages = loadMessages;

        window.updateMessage = async (key) => {
            const textarea = document.getElementById(`msg-val-${key}`);
            const result = await apiRequest('update_message', { message_key: key, message_value: textarea.value });
            if (result.success) {
                textarea.classList.add('!bg-green-500/20', '!border-green-500');
                setTimeout(() => textarea.classList.remove('!bg-green-500/20', '!border-green-500'), 2000);
            } else {
                alert('خطا در بروزرسانی پیام.');
                textarea.classList.add('!bg-red-500/20', '!border-red-500');
                setTimeout(() => textarea.classList.remove('!bg-red-500/20', '!border-red-500'), 2000);
        };
        
        document.getElementById('help-variables-btn').addEventListener('click', () => {
             openModal(`
                <div class="flex justify-between items-center border-b border-slate-600 pb-3 mb-4">
                    <h3 class="text-xl font-bold">راهنمای متغیرهای پیام</h3>
                    <button onclick="closeModal()" class="text-2xl">&times;</button>
                </div>
                <div class="max-h-[70vh] overflow-y-auto pr-2 space-y-3 text-sm">
                    <p><code>{service_name}</code> - نام نمایشی سرویس</p>
                    <p><code>{new_name}</code> - نام جدید سرویس (بعد از تغییر نام)</p>
                    <p><code>{location}</code> - نام لوکیشن یا سرور</p>
                    <p><code>{size}</code> - حجم کل سرویس (عدد)</p>
                    <p><code>{amount}</code> - قیمت سرویس (عدد فرمت شده)</p>
                    <p><code>{days}</code> - مدت زمان سرویس به روز</p>
                    <p><code>{date}</code> - تاریخ انقضای سرویس (شمسی)</p>
                    <p><code>{status}</code> - وضعیت فعلی سرویس (فعال/غیرفعال)</p>
                    <p><code>{used}</code> - حجم مصرف شده سرویس</p>
                    <p><code>{total}</code> - حجم کل سرویس (مشابه size)</p>
                    <p><code>{percent}</code> - درصد حجم مصرف شده</p>
                    <p><code>{progress_bar}</code> - نوار گرافیکی حجم مصرفی</p>
                    <p><code>{user_id}</code> - آیدی عددی کاربر</p>
                    <p><code>{service_count}</code> - تعداد کل سرویس‌های کاربر</p>
                    <p><code>{order_code}</code> - کد سفارش برای پرداخت</p>
                    <p><code>{card_info}</code> - اطلاعات شماره کارت</p>
                </div>`);
        });

        // --- Broadcast Actions ---
        document.getElementById('send-broadcast-btn').addEventListener('click', async () => {
            const message = document.getElementById('broadcast-message').value;
            if (!message.trim()) { alert('لطفا متن پیام را وارد کنید.'); return; }
            if (!confirm('آیا از ارسال این پیام به تمام کاربران مطمئن هستید؟')) return;
            const statusDiv = document.getElementById('broadcast-status');
            statusDiv.innerHTML = '<p class="text-blue-400 font-medium">در حال آماده‌سازی برای ارسال...</p>';
            const result = await apiRequest('broadcast', { message: message });
            if (result.success) {
                statusDiv.innerHTML = '<p class="text-green-400 font-medium">پیام در صف ارسال قرار گرفت.</p>';
                document.getElementById('broadcast-message').value = '';
            } else {
                statusDiv.innerHTML = `<p class="text-red-400 font-medium">خطا: ${result.message}</p>`;
        });

        // --- Settings Actions ---
        async function loadSettings() {
            const data = await apiRequest('get_settings');
            if (data.success) {
                document.getElementById('admin-id-input').value = data.settings.admin_id || '';
                document.getElementById('card-payment-status-toggle').checked = data.settings.card_payment_status === 'on';
                document.getElementById('card-number-input').value = data.settings.card_number || '';
                document.getElementById('crypto-payment-status-toggle').checked = data.settings.crypto_payment_status === 'on';
                document.getElementById('crypto-wallets-input').value = data.settings.crypto_wallets || '';
                document.getElementById('kyc-status-toggle').checked = data.settings.kyc_status === 'on';
                document.getElementById('kyc-channel-id-input').value = data.settings.kyc_channel_id || '';
                document.getElementById('kyc-message-input').value = data.settings.kyc_message || '';
                document.getElementById('join-status-toggle').checked = data.settings.join_status === 'on';
                document.getElementById('join-channels-input').value = data.settings.join_channels || '';
                document.getElementById('bot-status-toggle').checked = data.settings.bot_status === 'on';
                // Update image toggle
                const imgToggle = document.getElementById('service-image-status-toggle');
                if (imgToggle) {
                    imgToggle.checked = (data.settings.service_image_enabled === 'on');
                    // Ensure UI enable/disable state syncs when set programmatically
                    imgToggle.dispatchEvent(new Event('change'));
                // Update preview if image template exists
                const _prev = document.getElementById('service-template-preview');
                if (_prev) {
                    if (data.settings.service_template_url) {
                        _prev.src = data.settings.service_template_url;
                        _prev.classList.remove('hidden');
                    } else {
                        _prev.src = '';
                        _prev.classList.add('hidden');
                // Payment image toggle
                const payToggle = document.getElementById('payment-image-status-toggle');
                if (payToggle) {
                    payToggle.checked = (data.settings.payment_image_enabled === 'on');
                    payToggle.dispatchEvent(new Event('change'));
                const payPrev = document.getElementById('payment-template-preview');
                if (payPrev) {
                    if (data.settings.payment_template_url) {
                        payPrev.src = data.settings.payment_template_url;
                        payPrev.classList.remove('hidden');
                    } else {
                        payPrev.src = '';
                        payPrev.classList.add('hidden');
                // Sync payments details visibility
                try { updatePaymentSectionsVisibility(); } catch(e){}
        window.loadSettings = loadSettings;
        
        document.getElementById('save-settings-btn').addEventListener('click', async () => {
            const settings = {
                admin_id: document.getElementById('admin-id-input').value,
                card_payment_status: document.getElementById('card-payment-status-toggle').checked ? 'on' : 'off',
                card_number: document.getElementById('card-number-input').value,
                crypto_payment_status: document.getElementById('crypto-payment-status-toggle').checked ? 'on' : 'off',
                crypto_wallets: document.getElementById('crypto-wallets-input').value,
                kyc_status: document.getElementById('kyc-status-toggle').checked ? 'on' : 'off',
                kyc_channel_id: document.getElementById('kyc-channel-id-input').value,
                kyc_message: document.getElementById('kyc-message-input').value,
                join_status: document.getElementById('join-status-toggle').checked ? 'on' : 'off',
                join_channels: document.getElementById('join-channels-input').value,
                bot_status: document.getElementById('bot-status-toggle').checked ? 'on' : 'off',
                service_image_enabled: (document.getElementById('service-image-status-toggle')?.checked ? 'on' : 'off'),
                payment_image_enabled: (document.getElementById('payment-image-status-toggle')?.checked ? 'on' : 'off')
            };
            const result = await apiRequest('update_settings', settings);
            alert(result.success ? 'تنظیمات با موفقیت ذخیره شد.' : 'خطا: ' + result.message);
        });

        // Inject image template upload UI (once) and wire events
        (function __injectTemplateUI(){
            const container = document.getElementById('settings-cards');
            if (!container) return;
            if (document.getElementById('service-template-file')) return; // already exists
            // Ensure images group wrapper exists
            let imagesGroup = document.getElementById('images-group');
            if (!imagesGroup) {
                imagesGroup = document.createElement('div');
                imagesGroup.id = 'images-group';
                imagesGroup.className = 'border-t border-slate-600 pt-6';
                imagesGroup.innerHTML = `
                  <div class="rounded-xl bg-white/5 border border-white/10 p-4 md:p-5">
                    <div class="flex items-center gap-3 mb-4">
                      <i class="fas fa-images text-indigo-300"></i>
                      <h3 class="font-bold">بخش تصاویر</h3>
                    </div>
                    <div id="images-grid" class="grid grid-cols-1 lg:grid-cols-2 gap-4"></div>
                  </div>`;
                container.appendChild(imagesGroup);
            const imagesGrid = document.getElementById('images-grid') || imagesGroup;
            const block = document.createElement('div');
            block.className = '';
            block.innerHTML = `
                <div class="rounded-xl bg-white/5 border border-white/10 p-4 md:p-5">
                  <div class="flex items-center justify-between mb-4">
                    <div class="flex items-center gap-3">
                      <i class="fas fa-image text-indigo-300"></i>
                      <h3 class="font-bold">تصویر قالب کارت سرویس</h3>
                    </div>
                    <label class="flex items-center cursor-pointer text-sm">
                      <input type="checkbox" id="service-image-status-toggle" class="hidden peer">
                      <span class="toggle-track w-12 h-6 rounded-full bg-gray-600 peer-checked:bg-green-500 transition-all duration-300 relative"></span>
                      <span class="mr-2">ارسال تصویر</span>
                    </label>
                  </div>
                  <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <div class="md:col-span-2">
                      <div id="template-drop-zone" class="flex items-center justify-between gap-3 border-2 border-dashed border-slate-500/40 rounded-lg p-4 hover:border-indigo-400/60 transition cursor-pointer bg-white/5">
                        <div class="flex items-center gap-3">
                          <i class="fas fa-upload text-slate-400"></i>
                          <div>
                            <p class="font-medium">تصویر را انتخاب کنید یا بکشید و رها کنید</p>
                            <p class="text-xs text-slate-400">حداکثر ۵ مگابایت • PNG/JPG/WEBP • ترجیحاً مربع (۱:۱)</p>
                          </div>
                        </div>
                        <button type="button" id="choose-template-btn" class="liquid-pill px-3 py-2 text-sm">انتخاب فایل</button>
                      </div>
                      <input type="file" id="service-template-file" accept="image/png,image/jpeg,image/webp" class="hidden" />
                      <div class="mt-3 flex gap-3">
                        <button type="button" id="upload-template-btn" class="btn-accent btn-sm">آپلود تصویر</button>
                        <button type="button" id="remove-template-btn" class="btn-outline-danger btn-sm">حذف</button>
                      </div>
                    </div>
                    <div class="flex items-center md:justify-end">
                      <img id="service-template-preview" src="" alt="Service Template" class="w-36 h-36 object-cover rounded-lg border border-white/10 bg-slate-800/50 hidden" />
                    </div>
                  </div>
                </div>`;
            (imagesGrid || container).appendChild(block);

            const fileInput = document.getElementById('service-template-file');
            const dropZone  = document.getElementById('template-drop-zone');
            const chooseBtn = document.getElementById('choose-template-btn');
            const prev      = document.getElementById('service-template-preview');
            const uploadBtn = document.getElementById('upload-template-btn');
            const removeBtn = document.getElementById('remove-template-btn');
            const imgToggle = document.getElementById('service-image-status-toggle');

            function setEnabledControls(enabled){
                const dis = enabled ? false : true;
                [uploadBtn, removeBtn, chooseBtn, fileInput].forEach(el => { if (el) el.disabled = dis; });
                dropZone.classList.toggle('opacity-50', !enabled);
                dropZone.classList.toggle('pointer-events-none', !enabled);
            function previewLocalFile(file){
                if (!file) return;
                const reader = new FileReader();
                reader.onload = () => { prev.src = reader.result; prev.classList.remove('hidden'); };
                reader.readAsDataURL(file);
            chooseBtn?.addEventListener('click', () => fileInput.click());
            dropZone?.addEventListener('click', () => fileInput.click());
            dropZone?.addEventListener('dragover', (e) => { e.preventDefault(); dropZone.classList.add('ring-2','ring-indigo-400/50'); });
            dropZone?.addEventListener('dragleave', () => { dropZone.classList.remove('ring-2','ring-indigo-400/50'); });
            dropZone?.addEventListener('drop', (e) => {
                e.preventDefault();
                dropZone.classList.remove('ring-2','ring-indigo-400/50');
                const f = e.dataTransfer?.files?.[0];
                if (f) { fileInput.files = e.dataTransfer.files; previewLocalFile(f); }
            });
            fileInput?.addEventListener('change', () => { const f = fileInput.files?.[0]; if (f) previewLocalFile(f); });
            imgToggle?.addEventListener('change', () => setEnabledControls(imgToggle.checked));
            setEnabledControls(imgToggle?.checked ?? true);

            document.getElementById('upload-template-btn')?.addEventListener('click', async () => {
                if (!fileInput.files || !fileInput.files[0]) { alert('لطفاً فایل تصویر را انتخاب کنید.'); return; }
                const fd = new FormData();
                fd.append('action','upload_service_template');
                fd.append('file', fileInput.files[0]);
                try {
                    const resp = await fetch('api.php', { method:'POST', body: fd, credentials: 'same-origin', cache: 'no-store' });
                    const data = await resp.json();
                    console.debug('upload_service_template response', data);
                    if (data.success) {
                        prev.src = data.url;
                        prev.classList.remove('hidden');
                        alert('تصویر با موفقیت آپلود شد.');
                    } else {
                        // If limited by upload_max_filesize, try base64 fallback with client-side compress
                        const details = (data.hint || '') + ' ' + (data.message || '');
                        if (/upload_max_filesize/i.test(details)) {
                            if (confirm('محدودیت upload_max_filesize اجازه آپلود مستقیم نمی‌دهد. تلاش برای فشرده‌سازی و آپلود جایگزین؟')) {
                                try {
                                    const ok = await compressAndUploadBase64(fileInput.files[0]);
                                    if (ok) return; // success path handled inside
                                } catch (e) { console.error(e); }
                        alert((data.message || 'خطا در آپلود تصویر') + (data.hint ? ('\n' + data.hint) : ''));
                } catch(e) { alert('خطای شبکه/سرور در آپلود.'); }
            });

            document.getElementById('remove-template-btn')?.addEventListener('click', async () => {
                if (!confirm('تصویر قالب حذف شود و از پیش‌فرض استفاده گردد؟')) return;
                const fd = new FormData(); fd.append('action','delete_service_template');
                try {
                    const resp = await fetch('api.php', { method:'POST', body: fd, credentials: 'same-origin', cache: 'no-store' });
                    const data = await resp.json();
                    if (data.success) {
                        const prev = document.getElementById('service-template-preview');
                        prev.src = '';
                        prev.classList.add('hidden');
                        const fi = document.getElementById('service-template-file'); if (fi) fi.value = '';
                        alert('قالب به حالت پیش‌فرض بازگردانده شد.');
                    } else { alert(data.message || 'خطا در حذف تصویر'); }
                } catch(e) { alert('خطای شبکه/سرور.'); }
            });
        })();

        // Inject payment thumbnail upload UI
        (function __injectPaymentTemplateUI(){
            const container = document.getElementById('settings-cards');
            if (!container) return;
            if (document.getElementById('payment-template-file')) return; // already exists
            // Ensure images group wrapper exists
            let imagesGroup = document.getElementById('images-group');
            if (!imagesGroup) {
                imagesGroup = document.createElement('div');
                imagesGroup.id = 'images-group';
                imagesGroup.className = 'border-t border-slate-600 pt-6';
                imagesGroup.innerHTML = `
                  <div class=\"rounded-xl bg-white/5 border border-white/10 p-4 md:p-5\">\n                    <div class=\"flex items-center gap-3 mb-4\">\n                      <i class=\"fas fa-images text-indigo-300\"></i>\n                      <h3 class=\"font-bold\">بخش تصاویر</h3>\n                    </div>\n                    <div id=\"images-grid\" class=\"grid grid-cols-1 lg:grid-cols-2 gap-4\"></div>\n                  </div>`;
                container.appendChild(imagesGroup);
            const imagesGrid = document.getElementById('images-grid') || imagesGroup;
            const block = document.createElement('div');
            block.className = '';
            block.innerHTML = `
                <div class="rounded-xl bg-white/5 border border-white/10 p-4 md:p-5">
                  <div class="flex items-center justify-between mb-4">
                    <div class="flex items-center gap-3">
                      <i class="fas fa-image text-pink-300"></i>
                      <h3 class="font-bold">تصویر بندانگشتی پرداخت</h3>
                    </div>
                    <label class="flex items-center cursor-pointer text-sm">
                      <input type="checkbox" id="payment-image-status-toggle" class="hidden peer">
                      <span class="toggle-track w-12 h-6 rounded-full bg-gray-600 peer-checked:bg-green-500 transition-all duration-300 relative"></span>
                      <span class="mr-2">ارسال تصویر</span>
                    </label>
                  </div>
                  <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <div class="md:col-span-2">
                      <div id="payment-template-drop-zone" class="flex items-center justify-between gap-3 border-2 border-dashed border-slate-500/40 rounded-lg p-4 hover:border-indigo-400/60 transition cursor-pointer bg-white/5">
                        <div class="flex items-center gap-3">
                          <i class="fas fa-upload text-slate-400"></i>
                          <div>
                            <p class="font-medium">تصویر را انتخاب کنید یا بکشید و رها کنید</p>
                            <p class="text-xs text-slate-400">حداکثر ۵ مگابایت • PNG/JPG/WEBP • ترجیحاً مربع (۱:۱)</p>
                          </div>
                        </div>
                        <button type="button" id="choose-payment-template-btn" class="liquid-pill px-3 py-2 text-sm">انتخاب فایل</button>
                      </div>
                      <input type="file" id="payment-template-file" accept="image/png,image/jpeg,image/webp" class="hidden" />
                      <div class="mt-3 flex gap-3">
                        <button type="button" id="upload-payment-template-btn" class="btn-accent btn-sm">آپلود تصویر</button>
                        <button type="button" id="remove-payment-template-btn" class="btn-outline-danger btn-sm">حذف</button>
                      </div>
                    </div>
                    <div class="flex items-center md:justify-end">
                      <img id="payment-template-preview" src="" alt="Payment Template" class="w-36 h-36 object-cover rounded-lg border border-white/10 bg-slate-800/50 hidden" />
                    </div>
                  </div>
                </div>`;
            (imagesGrid || container).appendChild(block);

            const fileInput = document.getElementById('payment-template-file');
            const dropZone  = document.getElementById('payment-template-drop-zone');
            const chooseBtn = document.getElementById('choose-payment-template-btn');
            const prev      = document.getElementById('payment-template-preview');
            const uploadBtn = document.getElementById('upload-payment-template-btn');
            const removeBtn = document.getElementById('remove-payment-template-btn');
            const imgToggle = document.getElementById('payment-image-status-toggle');

            function setEnabledControls(enabled){
                const dis = enabled ? false : true;
                [uploadBtn, removeBtn, chooseBtn, fileInput].forEach(el => { if (el) el.disabled = dis; });
                dropZone.classList.toggle('opacity-50', !enabled);
                dropZone.classList.toggle('pointer-events-none', !enabled);
            }
            function previewLocalFile(file){
                if (!file) return;
                const reader = new FileReader();
                reader.onload = e => { prev.src = e.target.result; prev.classList.remove('hidden'); };
                reader.readAsDataURL(file);
            }
            chooseBtn?.addEventListener('click', () => fileInput.click());
            dropZone?.addEventListener('click', () => fileInput.click());
            dropZone?.addEventListener('dragover', (e) => { e.preventDefault(); dropZone.classList.add('ring-2','ring-indigo-400/50'); });
            dropZone?.addEventListener('dragleave', () => { dropZone.classList.remove('ring-2','ring-indigo-400/50'); });
            dropZone?.addEventListener('drop', (e) => { e.preventDefault(); dropZone.classList.remove('ring-2','ring-indigo-400/50'); const f = e.dataTransfer?.files?.[0]; if (f) { fileInput.files = e.dataTransfer.files; previewLocalFile(f); } });
            fileInput?.addEventListener('change', () => { const f = fileInput.files?.[0]; if (f) previewLocalFile(f); });
            imgToggle?.addEventListener('change', () => setEnabledControls(imgToggle.checked));
            setEnabledControls(imgToggle?.checked ?? true);

            document.getElementById('upload-payment-template-btn')?.addEventListener('click', async () => {
                if (!fileInput.files || !fileInput.files[0]) { alert('لطفاً فایل تصویر را انتخاب کنید.'); return; }
                const fd = new FormData();
                fd.append('action','upload_payment_template');
                fd.append('file', fileInput.files[0]);
                try {
                    const resp = await fetch('api.php', { method:'POST', body: fd, credentials: 'same-origin', cache: 'no-store' });
                    const data = await resp.json();
                    if (data.success) { prev.src = data.url; prev.classList.remove('hidden'); alert('تصویر با موفقیت آپلود شد.'); }
                    else {
                        const details = (data.hint || '') + ' ' + (data.message || '');
                        if (/upload_max_filesize/i.test(details)) {
                            if (confirm('محدودیت upload_max_filesize اجازه آپلود مستقیم نمی‌دهد. تلاش برای فشرده‌سازی و آپلود جایگزین؟')) { try { const ok = await compressAndUploadPaymentBase64(fileInput.files[0]); if (ok) return; } catch(e){} }
                        alert((data.message || 'خطا در آپلود تصویر') + (data.hint ? ('\n' + data.hint) : ''));
                } catch(e) { alert('خطای شبکه/سرور در آپلود.'); }
            });

            document.getElementById('remove-payment-template-btn')?.addEventListener('click', async () => {
                if (!confirm('تصویر حذف شود و از پیش‌فرض استفاده گردد؟')) return;
                const fd = new FormData(); fd.append('action','delete_payment_template');
                try {
                    const resp = await fetch('api.php', { method:'POST', body: fd, credentials: 'same-origin', cache: 'no-store' });
                    const data = await resp.json();
                    if (data.success) { prev.src=''; prev.classList.add('hidden'); const fi = document.getElementById('payment-template-file'); if (fi) fi.value=''; alert('تصویر حذف شد.'); }
                    else { alert(data.message || 'خطا در حذف تصویر'); }
                } catch(e) { alert('خطای شبکه/سرور.'); }
            });

            async function compressAndUploadPaymentBase64(file) {
                const dataUrl = await new Promise((resolve, reject) => { const r = new FileReader(); r.onerror=()=>reject('file read error'); r.onload=()=>resolve(r.result); r.readAsDataURL(file); });
                const img = await new Promise((resolve, reject) => { const im = new Image(); im.onload=()=>resolve(im); im.onerror=()=>reject('image load error'); im.src=dataUrl; });
                const maxDim = 1200; let {width:w,height:h}=img; if (w>h && w>maxDim){h=Math.round(h*(maxDim/w)); w=maxDim;} else if(h>w && h>maxDim){w=Math.round(w*(maxDim/h)); h=maxDim;} else if(w===h && w>maxDim){w=h=maxDim;}
                const canvas=document.createElement('canvas'); canvas.width=w; canvas.height=h; const ctx=canvas.getContext('2d'); ctx.drawImage(img,0,0,w,h);
                let outUrl = canvas.toDataURL('image/webp',0.92); if (outUrl.length>7.5*1024*1024){ outUrl = canvas.toDataURL('image/jpeg',0.85);} 
                const params = new URLSearchParams(); params.set('action','upload_payment_template_b64'); params.set('data', outUrl);
                const resp = await fetch('api.php', { method:'POST', body: params, credentials:'same-origin', cache:'no-store', headers:{'Content-Type':'application/x-www-form-urlencoded'} });
                const data = await resp.json(); if (data.success){ prev.src=data.url; prev.classList.remove('hidden'); alert('تصویر پس از فشرده‌سازی آپلود شد.'); return true;} else { alert((data.message||'خطا در آپلود (base64)')); return false; }
        })();

        // Client-side compress + base64 fallback
        async function compressAndUploadBase64(file) {
            const dataUrl = await new Promise((resolve, reject) => {
                const r = new FileReader();
                r.onerror = () => reject('file read error');
                r.onload = () => resolve(r.result);
                r.readAsDataURL(file);
            });
            const img = await new Promise((resolve, reject) => {
                const im = new Image();
                im.onload = () => resolve(im);
                im.onerror = () => reject('image load error');
                im.src = dataUrl;
            });
            // Resize to max 1200px (keeping aspect) to reduce size
            const maxDim = 1200;
            let { width: w, height: h } = img;
            if (w > h && w > maxDim) { h = Math.round(h * (maxDim / w)); w = maxDim; }
            else if (h > w && h > maxDim) { w = Math.round(w * (maxDim / h)); h = maxDim; }
            else if (w === h && w > maxDim) { w = h = maxDim; }

            const canvas = document.createElement('canvas');
            canvas.width = w; canvas.height = h;
            const ctx = canvas.getContext('2d');
            ctx.drawImage(img, 0, 0, w, h);

            // Prefer WEBP for smaller files if supported
            const quality = 0.92;
            let outUrl = canvas.toDataURL('image/webp', quality);
            if (outUrl.length > 7.5 * 1024 * 1024) { // still too big in post (approx)
                outUrl = canvas.toDataURL('image/jpeg', 0.85);

            const params = new URLSearchParams();
            params.set('action', 'upload_service_template_b64');
            params.set('data', outUrl);
            const resp = await fetch('api.php', { method: 'POST', body: params, credentials: 'same-origin', cache: 'no-store', headers: { 'Content-Type': 'application/x-www-form-urlencoded' } });
            const data = await resp.json();
            if (data.success) {
                const prev = document.getElementById('service-template-preview');
                prev.src = data.url;
                prev.classList.remove('hidden');
                alert('تصویر پس از فشرده‌سازی آپلود شد.');
                return true;
            } else {
                alert((data.message || 'خطا در آپلود (base64)'));
                return false;

        // =================================================================
        // ===== INITIAL LOAD =====
        // =================================================================
        loadDashboard();
    });
    </script>
    <script src="assets/admin.js?v=drawer_filters_fix"></script>

</body>
</html>







